<?php
/**
 * Health checks for FreshNews.ai Settings
 *
 * @package FreshNews_AI_Content_Hub
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Parse Signals base URL from plugin options.
 * Returns empty string if not configured or invalid.
 *
 * @return string Base URL (no trailing slash) or empty.
 */
function fn_parse_signals_base() {
	return function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
}

/**
 * Perform HTTP GET request for health checks.
 *
 * @param string $url URL to fetch.
 * @return array{ ok: bool, status: int|null, body: string, error: string|null }
 */
function fn_http_check( $url ) {
	$res = wp_remote_get(
		$url,
		array(
			'timeout'    => 6,
			'redirection' => 3,
			'sslverify'  => true,
			'user-agent' => 'FreshNews-AI-Content-Hub-Plugin/' . ( defined( 'FRESHNEWS_AI_CONTENT_HUB_VERSION' ) ? FRESHNEWS_AI_CONTENT_HUB_VERSION : '1.0.0' ),
		)
	);
	if ( is_wp_error( $res ) ) {
		return array( 'ok' => false, 'status' => null, 'body' => '', 'error' => $res->get_error_message() );
	}
	$status = wp_remote_retrieve_response_code( $res );
	$body   = wp_remote_retrieve_body( $res );
	return array( 'ok' => $status >= 200 && $status < 400, 'status' => $status, 'body' => $body, 'error' => null );
}

/**
 * Perform DNS check on host.
 * Returns green if CNAME to ingest.freshnews.ai or *.freshnews.ai;
 * yellow if no CNAME but A/AAAA exists (DNS flattening);
 * red if no records or lookup fails.
 *
 * @param string $host Hostname (e.g. signals.customer.com).
 * @return array{ status: 'green'|'yellow'|'red', message: string }
 */
function fn_dns_check( $host ) {
	$host = preg_replace( '/[^a-z0-9.-]/i', '', $host );
	if ( empty( $host ) ) {
		return array( 'status' => 'red', 'message' => __( 'Invalid hostname', 'freshnews-ai-content-hub' ) );
	}

	$cname = @dns_get_record( $host, DNS_CNAME );
	$a_aaaa = false;
	if ( defined( 'DNS_A' ) && defined( 'DNS_AAAA' ) ) {
		$a_aaaa = @dns_get_record( $host, DNS_A | DNS_AAAA );
	} else {
		$a_aaaa = @dns_get_record( $host, DNS_A );
		if ( empty( $a_aaaa ) && defined( 'DNS_AAAA' ) ) {
			$a_aaaa = @dns_get_record( $host, DNS_AAAA );
		}
	}

	if ( false === $cname ) {
		$cname = array();
	}
	if ( false === $a_aaaa ) {
		$a_aaaa = array();
	}

	// Green: CNAME exists and target is ingest.freshnews.ai or *.freshnews.ai
	if ( ! empty( $cname ) && isset( $cname[0]['target'] ) ) {
		$target = strtolower( rtrim( $cname[0]['target'], '.' ) );
		if ( $target === 'ingest.freshnews.ai' || substr( $target, -strlen( '.freshnews.ai' ) ) === '.freshnews.ai' ) {
			return array( 'status' => 'green', 'message' => __( 'CNAME configured correctly', 'freshnews-ai-content-hub' ) );
		}
		return array( 'status' => 'red', 'message' => __( 'CNAME target not FreshNews.ai', 'freshnews-ai-content-hub' ) );
	}

	// Yellow: no CNAME but A/AAAA exists (DNS flattening) - caller must verify HTTP checks pass
	if ( ! empty( $a_aaaa ) ) {
		return array( 'status' => 'yellow', 'message' => __( 'A/AAAA found (DNS flattening)', 'freshnews-ai-content-hub' ) );
	}

	return array( 'status' => 'red', 'message' => __( 'No CNAME or A/AAAA records', 'freshnews-ai-content-hub' ) );
}

/**
 * Get test language from plugin settings (first 2 letters, default "en").
 *
 * @return string Two-letter language code.
 */
function fn_get_test_lang() {
	$options = get_option( 'freshnews_ai_newsroom_options', array() );
	$lang    = isset( $options['language'] ) ? trim( (string) $options['language'] ) : 'en';
	if ( empty( $lang ) ) {
		return 'en';
	}
	$lang = substr( preg_replace( '/[^a-z-]/i', '', $lang ), 0, 2 );
	return empty( $lang ) ? 'en' : strtolower( $lang );
}

/**
 * Transient key for health checks. Keyed by site URL + signals base.
 *
 * @param string $signals_base Signals base URL.
 * @return string Transient key.
 */
function fn_health_checks_transient_key( $signals_base ) {
	return 'freshnews_health_checks_' . md5( home_url() . '|' . $signals_base );
}

/**
 * Run all health checks and return results.
 *
 * @param string $signals_base Signals base URL (no trailing slash).
 * @return array<string, array{ status: string, message: string }>
 */
function fn_run_health_checks( $signals_base ) {
	$signals_base = untrailingslashit( $signals_base );
	$results     = array();

	// 1) DNS / CNAME check
	$parsed = wp_parse_url( $signals_base );
	$host   = isset( $parsed['host'] ) ? $parsed['host'] : '';
	$dns    = fn_dns_check( $host );
	$results['dns'] = array( 'status' => $dns['status'], 'message' => $dns['message'] );

	// If DNS is red, we can still run HTTP checks for diagnostics, but DNS stays red.
	// If DNS is yellow, we need HTTP checks to pass to keep yellow; otherwise treat as red.
	$http_ok = false;

	// 2) Hub reachability
	$hub = fn_http_check( $signals_base . '/' );
	if ( $hub['ok'] && in_array( (int) $hub['status'], array( 200, 301, 302 ), true ) ) {
		$results['hub'] = array( 'status' => 'green', 'message' => __( 'Reachable', 'freshnews-ai-content-hub' ) );
		$http_ok        = true;
	} else {
		$err = $hub['error'] ?: ( 'HTTP ' . ( $hub['status'] ?: 'error' ) );
		$results['hub'] = array( 'status' => 'red', 'message' => esc_html( $err ) );
	}

	// 3) Sitemap reachability/validity
	$sitemap = fn_http_check( $signals_base . '/sitemap.xml' );
	if ( $sitemap['ok'] && (int) $sitemap['status'] === 200 ) {
		$body = $sitemap['body'];
		$valid = ( strpos( $body, '<urlset' ) !== false || strpos( $body, '<sitemapindex' ) !== false );
		if ( $valid ) {
			$results['sitemap'] = array( 'status' => 'green', 'message' => __( 'Valid sitemap', 'freshnews-ai-content-hub' ) );
			$http_ok            = true;
		} else {
			$results['sitemap'] = array( 'status' => 'red', 'message' => __( 'Invalid or empty sitemap', 'freshnews-ai-content-hub' ) );
		}
	} else {
		$err = $sitemap['error'] ?: ( 'HTTP ' . ( $sitemap['status'] ?: 'error' ) );
		$results['sitemap'] = array( 'status' => 'red', 'message' => esc_html( $err ) );
	}

	// 4) Latest page
	$lang   = fn_get_test_lang();
	$latest = fn_http_check( $signals_base . '/' . $lang . '/latest' );
	if ( $latest['ok'] && in_array( (int) $latest['status'], array( 200, 301, 302 ), true ) ) {
		$results['latest'] = array( 'status' => 'green', 'message' => __( 'Reachable', 'freshnews-ai-content-hub' ) );
		$http_ok           = true;
	} else {
		$err = $latest['error'] ?: ( 'HTTP ' . ( $latest['status'] ?: 'error' ) );
		$results['latest'] = array( 'status' => 'red', 'message' => esc_html( $err ) );
	}

	// 5) Footer embed script
	$footer = fn_http_check( $signals_base . '/embed/footer-pulse.min.js' );
	if ( $footer['ok'] && (int) $footer['status'] === 200 ) {
		$results['footer_script'] = array( 'status' => 'green', 'message' => __( 'Reachable', 'freshnews-ai-content-hub' ) );
		$http_ok                  = true;
	} else {
		$err = $footer['error'] ?: ( 'HTTP ' . ( $footer['status'] ?: 'error' ) );
		$results['footer_script'] = array( 'status' => 'red', 'message' => esc_html( $err ) );
	}

	// DNS yellow: only keep yellow if HTTP checks pass
	if ( $results['dns']['status'] === 'yellow' && $http_ok ) {
		$results['dns']['message'] = __( 'A/AAAA found (DNS flattening); HTTP checks pass', 'freshnews-ai-content-hub' );
	} elseif ( $results['dns']['status'] === 'yellow' ) {
		$results['dns'] = array( 'status' => 'red', 'message' => __( 'No CNAME; HTTP checks failed', 'freshnews-ai-content-hub' ) );
	}

	return $results;
}
