<?php
/**
 * Footer (Latest Signal) – hidden container + script for SEO/crawlers
 * Injected at end of wp_footer, invisible to users, readable by crawlers.
 *
 * @package FreshNews_AI_Content_Hub
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Get normalized Signals Hub base URL (no trailing slash).
 * Alias for freshnews_get_signals_url() for backward compatibility.
 *
 * @return string Empty if not configured or invalid.
 */
function freshnews_get_signals_base_url() {
	return function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
}

/**
 * Check if footer pulse integration is enabled.
 *
 * @return bool
 */
function freshnews_footer_pulse_enabled() {
	$options = get_option( 'freshnews_ai_newsroom_options', array() );
	if ( isset( $options['footer_pulse_enabled'] ) ) {
		return ! empty( $options['footer_pulse_enabled'] );
	}
	// Backward compat: old key was auto_footer_latest
	return ! empty( $options['auto_footer_latest'] );
}

/**
 * Get footer pulse label (default includes colon for script).
 *
 * @return string
 */
function freshnews_footer_pulse_label() {
	$options = get_option( 'freshnews_ai_newsroom_options', array() );
	$label   = isset( $options['footer_pulse_label'] ) ? trim( (string) $options['footer_pulse_label'] ) : '';
	if ( $label === '' ) {
		return 'Latest Signal: ';
	}
	return $label . ( substr( $label, -1 ) === ':' ? '' : ': ' );
}

/**
 * Get data-lang value for the script tag.
 * - auto: "auto"
 * - fixed: the configured lang code (e.g. en, he, en-US)
 *
 * @return string
 */
function freshnews_footer_pulse_data_lang() {
	$options = get_option( 'freshnews_ai_newsroom_options', array() );
	$mode    = isset( $options['footer_pulse_lang_mode'] ) ? $options['footer_pulse_lang_mode'] : 'auto';
	if ( $mode === 'fixed' && ! empty( $options['footer_pulse_lang'] ) ) {
		return sanitize_text_field( trim( (string) $options['footer_pulse_lang'] ) );
	}
	return 'auto';
}

/**
 * Resolve language for noscript fallback link (when mode=auto, use page/lang).
 *
 * @return string
 */
function freshnews_footer_pulse_resolve_lang() {
	$options = get_option( 'freshnews_ai_newsroom_options', array() );
	$mode    = isset( $options['footer_pulse_lang_mode'] ) ? $options['footer_pulse_lang_mode'] : 'auto';
	if ( $mode === 'fixed' && ! empty( $options['footer_pulse_lang'] ) ) {
		return sanitize_text_field( trim( (string) $options['footer_pulse_lang'] ) );
	}
	// auto: detect from WPML/Polylang or plugin language
	if ( function_exists( 'pll_current_language' ) ) {
		$lang = pll_current_language( 'slug' );
		if ( is_string( $lang ) && $lang !== '' ) {
			return strtolower( substr( $lang, 0, 2 ) ) === 'he' ? 'he' : ( strtolower( substr( $lang, 0, 2 ) ) === 'en' ? 'en' : $lang );
		}
	}
	$wpml = apply_filters( 'wpml_current_language', null );
	if ( is_string( $wpml ) && $wpml !== '' ) {
		$s = strtolower( substr( $wpml, 0, 2 ) );
		return $s === 'he' ? 'he' : ( $s === 'en' ? 'en' : $wpml );
	}
	$lang = isset( $options['language'] ) ? trim( (string) $options['language'] ) : 'en';
	return $lang !== '' ? strtolower( strtok( $lang, '-' ) ) : 'en';
}

/**
 * Render the hidden footer Latest Signal container + script.
 * Hooked to wp_footer with priority 9999.
 */
function freshnews_footer_pulse_render() {
	if ( is_admin() ) {
		return;
	}
	static $rendered = false;
	if ( $rendered ) {
		return;
	}
	if ( ! freshnews_footer_pulse_enabled() ) {
		return;
	}
	$signals_base = freshnews_get_signals_base_url();
	if ( $signals_base === '' ) {
		return;
	}
	$rendered = true;

	$lang   = freshnews_footer_pulse_resolve_lang();
	$label  = freshnews_footer_pulse_label();
	$label  = rtrim( $label, ': ' );
	$latest_url = $signals_base . '/' . $lang . '/latest';
	$label_colon = $label . ': ';
	?>
	<div id="freshnews-latest" class="freshnews-footer-pulse" style="position:absolute;left:-99999px;width:1px;height:1px;overflow:hidden;">
		<noscript>
			<a href="<?php echo esc_url( $latest_url ); ?>" target="_blank" rel="noopener noreferrer"><?php echo esc_html( $label ); ?></a>
		</noscript>
	</div>
	<script
		src="<?php echo esc_url( $signals_base . '/embed/footer-pulse.min.js' ); ?>"
		data-hub="<?php echo esc_attr( $signals_base ); ?>"
		data-container="#freshnews-latest"
		data-lang="<?php echo esc_attr( freshnews_footer_pulse_data_lang() ); ?>"
		data-label="<?php echo esc_attr( $label_colon ); ?>"
		data-target="_blank"
		defer
	></script>
	<?php
}
