<?php
/**
 * Main plugin class
 *
 * @package FreshNews_AI_Content_Hub
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main FreshNews AI Content Hub class
 */
class FreshNews_AI_Content_Hub {

	/**
	 * Flag to track if shortcode is present on the page
	 *
	 * @var bool
	 */
	private $shortcode_present = false;

	/**
	 * Current article slug being viewed
	 *
	 * @var string|null
	 */
	private $current_article_slug = null;

	/**
	 * Flag to track if FAQ shortcode is present on the page
	 *
	 * @var bool
	 */
	private $faq_shortcode_present = false;

	/**
	 * Initialize the plugin
	 */
	public function init() {
		// Register settings
		add_action( 'admin_init', array( $this, 'register_settings' ) );

		// Add admin menu
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );

		// Register shortcodes
		add_shortcode( 'freshnews_content_hub', array( $this, 'shortcode_handler' ) );
		add_shortcode( 'freshnews_newsroom', array( $this, 'shortcode_handler' ) ); // Backward compatibility
		add_shortcode( 'freshnews_faq', array( $this, 'shortcode_faq_handler' ) );

		// Enqueue scripts (when main shortcode or FAQ shortcode is present)
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );

		// Add canonical link to head
		add_action( 'wp_head', array( $this, 'add_canonical_link' ) );

		// Add JSON-LD schema for articles
		add_action( 'wp_head', array( $this, 'add_jsonld_schema' ), 5 );

		// Add CSS to hide page title and reduce whitespace
		add_action( 'wp_head', array( $this, 'add_hide_title_css' ), 20 );

		// Filter to remove page title when shortcode is present
		add_filter( 'the_title', array( $this, 'hide_page_title' ), 10, 2 );

		// Add body class when shortcode is present for more specific CSS targeting
		add_filter( 'body_class', array( $this, 'add_body_class' ) );

		// Add author block script (E-E-A-T optimization)
		add_action( 'wp_footer', array( $this, 'add_author_block_script' ) );

		// Auto-create/update page on settings save
		add_action( 'admin_init', array( $this, 'maybe_auto_create_page' ) );

		// Also trigger page creation when options are updated
		add_action( 'update_option_freshnews_ai_newsroom_options', array( $this, 'maybe_auto_create_page_on_save' ), 10, 2 );

		// Invalidate robots.txt check when settings change
		add_action( 'update_option_freshnews_ai_newsroom_options', array( $this, 'invalidate_robots_txt_check' ) );
		// Invalidate llms.txt check when settings change
		add_action( 'update_option_freshnews_ai_newsroom_options', array( $this, 'invalidate_llms_txt_check' ) );
		// Flush rewrite rules when llms.txt setting is toggled
		add_action( 'update_option_freshnews_ai_newsroom_options', array( $this, 'maybe_flush_rewrite_rules_on_llms_toggle' ), 10, 3 );
		// Invalidate footer pulse check when settings change
		add_action( 'update_option_freshnews_ai_newsroom_options', array( $this, 'invalidate_footer_pulse_check' ) );
		// Invalidate and re-run health checks when settings change
		add_action( 'update_option_freshnews_ai_newsroom_options', array( $this, 'invalidate_and_rerun_health_checks' ), 10, 2 );

		// Health checks admin-ajax
		add_action( 'wp_ajax_freshnews_run_health_checks', array( $this, 'ajax_run_health_checks' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_health_checks_script' ) );

		// Add admin notice if settings not configured (only for admins)
		add_action( 'admin_notices', array( $this, 'admin_notices' ) );

		// Add settings link to plugins page
		add_filter( 'plugin_action_links_' . FRESHNEWS_AI_CONTENT_HUB_PLUGIN_BASENAME, array( $this, 'add_plugin_action_links' ) );

		// Add rewrite rules for article URLs and llms.txt
		add_action( 'init', array( $this, 'add_rewrite_rules' ) );
		add_action( 'init', array( $this, 'maybe_flush_rewrite_rules_after_llms_toggle' ), 999 );
		add_filter( 'query_vars', array( $this, 'add_query_vars' ) );

		// Handle virtual endpoints (llms.txt before article URLs)
		add_action( 'template_redirect', array( $this, 'handle_llms_txt' ), 1 );
		// Handle article URLs early to prevent 404
		add_action( 'parse_request', array( $this, 'handle_article_url_early' ), 1 );
		add_action( 'template_redirect', array( $this, 'handle_article_url' ), 2 ); // Higher priority to run early

		// Add sitemap integration (WordPress 5.5+)
		if ( function_exists( 'wp_sitemaps_register_provider' ) ) {
			add_action( 'init', array( $this, 'register_sitemap_provider' ) );
		}

		// Register widget
		add_action( 'widgets_init', array( $this, 'register_widget' ) );

		// Append Signals Hub sitemap to robots.txt (virtual)
		add_filter( 'robots_txt', array( $this, 'append_sitemap_to_robots' ), 10, 2 );

		// Footer Latest Signal – auto-inject at wp_footer
		$footer_file = FRESHNEWS_AI_CONTENT_HUB_PLUGIN_DIR . 'includes/footer-latest.php';
		if ( file_exists( $footer_file ) ) {
			require_once $footer_file;
			add_action( 'wp_footer', 'freshnews_footer_pulse_render', 9999 );
		}
	}

	/**
	 * Get normalized Signals Hub base URL (no trailing slash, valid scheme).
	 *
	 * @return string Empty if not configured or invalid.
	 */
	public function get_signals_hub_base_url() {
		return function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
	}

	/**
	 * Append AI discovery rules and Sitemap lines to robots.txt output.
	 * Adds Allow for OAI-SearchBot and GPTBot (no duplicates), and Sitemap URLs.
	 *
	 * @param string $output Current robots.txt output.
	 * @param string $public Site visibility ( '1' or '0' ).
	 * @return string Modified output.
	 */
	public function append_sitemap_to_robots( $output, $public ) {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		if ( empty( $options['enable_robots_sitemaps'] ) ) {
			return $output;
		}
		$signals_base = $this->get_signals_hub_base_url();
		if ( empty( $signals_base ) ) {
			return $output;
		}
		$lines = array();

		// AI crawler Allow rules (only if not already present).
		if ( stripos( $output, 'OAI-SearchBot' ) === false ) {
			$lines[] = "User-agent: OAI-SearchBot\nAllow: /";
		}
		if ( stripos( $output, 'GPTBot' ) === false ) {
			$lines[] = "User-agent: GPTBot\nAllow: /";
		}

		// Sitemap lines (no duplicates).
		$base = rtrim( $signals_base );
		$main = 'Sitemap: ' . $base . '/sitemap.xml';
		$news = 'Sitemap: ' . $base . '/news-sitemap.xml';
		if ( stripos( $output, $main ) === false ) {
			$lines[] = $main;
		}
		if ( stripos( $output, $news ) === false ) {
			$lines[] = $news;
		}

		if ( empty( $lines ) ) {
			return $output;
		}
		return rtrim( $output ) . "\n\n" . implode( "\n\n", $lines ) . "\n";
	}

	/**
	 * Invalidate robots.txt and AI discovery status transients when settings are updated.
	 */
	public function invalidate_robots_txt_check() {
		delete_transient( self::ROBOTS_TXT_CHECK_TRANSIENT );
		delete_transient( self::AI_DISCOVERY_STATUS_TRANSIENT );
	}

	/**
	 * Invalidate llms.txt check transient when settings are updated.
	 */
	public function invalidate_llms_txt_check() {
		delete_transient( self::LLMS_TXT_CHECK_TRANSIENT );
	}

	/** Transient: flush rewrite rules on next request after llms.txt setting change (so rules are built with current options). */
	const FLUSH_REWRITE_AFTER_LLMS_TRANSIENT = 'freshnews_flush_rewrite_after_llms';

	/**
	 * When Enable llms.txt is toggled, schedule a flush on the next request so rewrite rules
	 * are regenerated with the current options (same-request flush would use stale in-memory rules).
	 *
	 * @param mixed  $old_value Old option value.
	 * @param mixed  $new_value New option value.
	 * @param string $option    Option name.
	 */
	public function maybe_flush_rewrite_rules_on_llms_toggle( $old_value, $new_value, $option ) {
		if ( $option !== 'freshnews_ai_newsroom_options' || ! is_array( $old_value ) || ! is_array( $new_value ) ) {
			return;
		}
		$old_llms = ! empty( $old_value['enable_llms_txt'] );
		$new_llms = ! empty( $new_value['enable_llms_txt'] );
		if ( $old_llms !== $new_llms ) {
			set_transient( self::FLUSH_REWRITE_AFTER_LLMS_TRANSIENT, 1, 60 );
		}
	}

	/**
	 * Flush rewrite rules on next request after llms.txt toggle (runs late on init so rules are already registered).
	 */
	public function maybe_flush_rewrite_rules_after_llms_toggle() {
		if ( ! get_transient( self::FLUSH_REWRITE_AFTER_LLMS_TRANSIENT ) ) {
			return;
		}
		delete_transient( self::FLUSH_REWRITE_AFTER_LLMS_TRANSIENT );
		flush_rewrite_rules();
	}

	/**
	 * Invalidate footer pulse check transient when settings are updated.
	 */
	public function invalidate_footer_pulse_check() {
		delete_transient( self::FOOTER_PULSE_CHECK_TRANSIENT );
	}

	/**
	 * Invalidate health checks transient and re-run tests on settings save.
	 *
	 * @param mixed $old_value Old option value.
	 * @param mixed $new_value New option value.
	 */
	public function invalidate_and_rerun_health_checks( $old_value, $new_value ) {
		if ( ! function_exists( 'fn_parse_signals_base' ) || ! function_exists( 'fn_run_health_checks' ) || ! function_exists( 'fn_health_checks_transient_key' ) ) {
			return;
		}
		// Option is already saved when this hook runs, so fn_parse_signals_base() gets the new value.
		$signals_base = fn_parse_signals_base();
		if ( empty( $signals_base ) ) {
			return;
		}
		$key = fn_health_checks_transient_key( $signals_base );
		delete_transient( $key );
		$results = fn_run_health_checks( $signals_base );
		set_transient( $key, $results, 10 * MINUTE_IN_SECONDS );
	}

	/**
	 * Enqueue health checks script on settings page.
	 *
	 * @param string $hook_suffix Current admin page.
	 */
	public function enqueue_health_checks_script( $hook_suffix ) {
		if ( 'settings_page_freshnews-ai-content-hub' !== $hook_suffix ) {
			return;
		}
		$script_url = FRESHNEWS_AI_CONTENT_HUB_PLUGIN_URL . 'assets/js/admin-health-checks.js';
		wp_enqueue_script(
			'freshnews-health-checks',
			$script_url,
			array( 'jquery' ),
			FRESHNEWS_AI_CONTENT_HUB_VERSION,
			true
		);
		wp_localize_script(
			'freshnews-health-checks',
			'freshnewsHealthChecks',
			array( 'ajaxurl' => admin_url( 'admin-ajax.php' ) )
		);
	}

	/**
	 * AJAX handler for Run health checks.
	 */
	public function ajax_run_health_checks() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized', 'freshnews-ai-content-hub' ) ), 403 );
		}
		$nonce = isset( $_POST['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'freshnews_run_health_checks' ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid nonce', 'freshnews-ai-content-hub' ) ), 403 );
		}
		if ( ! function_exists( 'fn_parse_signals_base' ) || ! function_exists( 'fn_run_health_checks' ) || ! function_exists( 'fn_health_checks_transient_key' ) ) {
			wp_send_json_error( array( 'message' => __( 'Health checks not available', 'freshnews-ai-content-hub' ) ) );
		}
		$signals_base = fn_parse_signals_base();
		if ( empty( $signals_base ) ) {
			wp_send_json_error( array( 'message' => __( 'Signals URL not configured', 'freshnews-ai-content-hub' ) ) );
		}
		$results = fn_run_health_checks( $signals_base );
		$key     = fn_health_checks_transient_key( $signals_base );
		set_transient( $key, $results, 10 * MINUTE_IN_SECONDS );
		wp_send_json_success( $results );
	}

	/**
	 * Render Health Checks section on settings page.
	 */
	public function render_health_checks_section() {
		if ( ! function_exists( 'fn_parse_signals_base' ) || ! function_exists( 'fn_run_health_checks' ) || ! function_exists( 'fn_health_checks_transient_key' ) ) {
			return;
		}
		$signals_base = fn_parse_signals_base();
		$results      = array();
		if ( ! empty( $signals_base ) ) {
			$key    = fn_health_checks_transient_key( $signals_base );
			$cached = get_transient( $key );
			if ( false !== $cached && is_array( $cached ) ) {
				$results = $cached;
			}
			// Never run tests on page load - only show cached results. User clicks "Run tests" to refresh.
		}
		$order  = array( 'dns', 'hub', 'sitemap', 'latest', 'footer_script' );
		$labels = array(
			'dns'          => __( 'DNS / CNAME', 'freshnews-ai-content-hub' ),
			'hub'          => __( 'Hub reachability', 'freshnews-ai-content-hub' ),
			'sitemap'      => __( 'Sitemap', 'freshnews-ai-content-hub' ),
			'latest'       => __( 'Latest page', 'freshnews-ai-content-hub' ),
			'footer_script' => __( 'Footer embed script', 'freshnews-ai-content-hub' ),
		);
		?>
		<div id="freshnews-health-checks-results" class="freshnews-health-checks-card" style="margin-top:24px;">
			<h2><?php esc_html_e( 'Health Checks', 'freshnews-ai-content-hub' ); ?></h2>
			<?php if ( empty( $signals_base ) ) : ?>
				<p class="description"><?php esc_html_e( 'Configure the Signals URL above to run health checks.', 'freshnews-ai-content-hub' ); ?></p>
			<?php else : ?>
				<p class="description"><?php echo esc_html( sprintf( __( 'Testing: %s', 'freshnews-ai-content-hub' ), $signals_base ) ); ?></p>
				<div class="freshnews-health-rows">
					<?php if ( empty( $results ) ) : ?>
						<p class="description"><?php esc_html_e( 'No cached results. Click "Run tests" to check.', 'freshnews-ai-content-hub' ); ?></p>
					<?php endif; ?>
					<?php foreach ( $order as $k ) : ?>
						<?php
						$item = isset( $results[ $k ] ) ? $results[ $k ] : null;
						if ( ! $item ) {
							continue;
						}
						$status = isset( $item['status'] ) ? sanitize_html_class( $item['status'] ) : 'red';
						$msg    = isset( $item['message'] ) ? esc_html( $item['message'] ) : '';
						$label  = isset( $labels[ $k ] ) ? $labels[ $k ] : $k;
						?>
						<div class="freshnews-health-row">
							<span class="freshnews-health-label"><?php echo esc_html( $label ); ?></span>
							<span class="freshnews-health-pill status-<?php echo esc_attr( $status ); ?>"><?php echo esc_html( $status ); ?></span>
							<span class="freshnews-health-msg"><?php echo $msg; ?></span>
						</div>
					<?php endforeach; ?>
				</div>
				<p style="margin-top:12px;">
					<button type="button" id="freshnews-run-health-tests" class="button">
						<?php esc_html_e( 'Run tests', 'freshnews-ai-content-hub' ); ?>
					</button>
					<span id="freshnews-health-tests-spinner" class="spinner" style="float:none;margin:0 0 0 8px;"></span>
				</p>
			<?php endif; ?>
		</div>
		<?php wp_nonce_field( 'freshnews_run_health_checks', 'freshnews_health_nonce', false ); ?>
		<style>
			.freshnews-health-checks-card { background:#fff; border:1px solid #c3c4c7; border-radius:4px; padding:16px 20px; box-shadow:0 1px 1px rgba(0,0,0,.04); }
			.freshnews-health-checks-card h2 { margin:0 0 12px 0; font-size:1.1em; }
			.freshnews-health-row { display:flex; align-items:center; gap:8px; margin:6px 0; }
			.freshnews-health-label { min-width:140px; font-weight:500; }
			.freshnews-health-pill { display:inline-block; padding:2px 8px; border-radius:4px; font-size:12px; text-transform:lowercase; }
			.freshnews-health-pill.status-green { background:#00a32a; color:#fff; }
			.freshnews-health-pill.status-yellow { background:#dba617; color:#1d2327; }
			.freshnews-health-pill.status-red { background:#d63638; color:#fff; }
			.freshnews-health-msg { color:#646970; font-size:13px; }
		</style>
		<?php
	}

	/**
	 * Transient key for footer pulse injection check.
	 */
	const FOOTER_PULSE_CHECK_TRANSIENT = 'freshnews_footer_pulse_check';

	/**
	 * Markers used to detect footer injection on homepage.
	 * Must match output from footer-latest.php.
	 */
	const FOOTER_PULSE_MARKER_CONTAINER = 'id="freshnews-latest"';
	const FOOTER_PULSE_MARKER_SCRIPT   = 'footer-pulse.min.js';

	/**
	 * Test if footer pulse is detected on homepage.
	 *
	 * @return array{ ok: bool, error: string|null }
	 */
	public function check_footer_pulse_injection() {
		$url = home_url( '/' );
		$res = wp_remote_get(
			$url,
			array(
				'timeout'     => 5,
				'sslverify'   => true,
				'user-agent'  => 'FreshNews-AI-Content-Hub-Plugin/' . FRESHNEWS_AI_CONTENT_HUB_VERSION,
			)
		);
		if ( is_wp_error( $res ) ) {
			return array( 'ok' => false, 'error' => $res->get_error_message() );
		}
		if ( wp_remote_retrieve_response_code( $res ) !== 200 ) {
			return array( 'ok' => false, 'error' => 'HTTP ' . wp_remote_retrieve_response_code( $res ) );
		}
		$body = wp_remote_retrieve_body( $res );
		$has_container = strpos( $body, self::FOOTER_PULSE_MARKER_CONTAINER ) !== false;
		$has_script    = strpos( $body, self::FOOTER_PULSE_MARKER_SCRIPT ) !== false;
		return array(
			'ok'    => $has_container && $has_script,
			'error' => null,
		);
	}

	/**
	 * Render footer pulse status banner on settings page.
	 */
	public function render_footer_pulse_status_banner() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		if ( ! freshnews_footer_pulse_enabled() ) {
			return;
		}
		$signals_base = $this->get_signals_hub_base_url();
		if ( empty( $signals_base ) ) {
			return;
		}
		$transient_key = self::FOOTER_PULSE_CHECK_TRANSIENT;
		$recheck_url   = admin_url( 'options-general.php?page=freshnews-ai-content-hub&freshnews_footer_recheck=1' );
		$recheck_url   = wp_nonce_url( $recheck_url, 'freshnews_footer_recheck', '_wpnonce' );

		if ( isset( $_GET['freshnews_footer_recheck'] ) && isset( $_GET['_wpnonce'] ) ) {
			if ( wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'freshnews_footer_recheck' ) ) {
				delete_transient( $transient_key );
			}
		}

		$cached = get_transient( $transient_key );
		if ( false !== $cached && is_array( $cached ) ) {
			$result = $cached;
		} else {
			$result = $this->check_footer_pulse_injection();
			set_transient( $transient_key, $result, MINUTE_IN_SECONDS );
		}

		if ( $result['ok'] ) {
			?>
			<div class="notice notice-success is-dismissible">
				<p>
					<strong><?php esc_html_e( 'Footer integration: Enabled & detected on homepage', 'freshnews-ai-content-hub' ); ?></strong>
					<a href="<?php echo esc_url( $recheck_url ); ?>"><?php esc_html_e( 'Re-check', 'freshnews-ai-content-hub' ); ?></a>
				</p>
			</div>
			<?php
		} else {
			?>
			<div class="notice notice-error is-dismissible">
				<p>
					<strong><?php esc_html_e( 'Footer integration: Enabled but not detected', 'freshnews-ai-content-hub' ); ?></strong>
				</p>
				<p>
					<?php esc_html_e( 'Possible causes: theme does not call wp_footer(), full-page caching, or a page builder replaces the footer.', 'freshnews-ai-content-hub' ); ?>
					<?php if ( ! empty( $result['error'] ) ) : ?>
						<?php echo esc_html( sprintf( __( 'Request error: %s', 'freshnews-ai-content-hub' ), $result['error'] ) ); ?>
					<?php endif; ?>
				</p>
				<p>
					<strong><?php esc_html_e( 'How to fix:', 'freshnews-ai-content-hub' ); ?></strong>
				</p>
				<ul style="list-style:disc;margin-left:20px;">
					<li><?php esc_html_e( 'Ensure your footer template includes wp_footer()', 'freshnews-ai-content-hub' ); ?> (<code>&lt;?php wp_footer(); ?&gt;</code>)</li>
					<li><?php esc_html_e( 'If you use caching (WP Rocket, Cloudflare, etc.), purge cache and re-test.', 'freshnews-ai-content-hub' ); ?></li>
				</ul>
				<p>
					<a href="<?php echo esc_url( $recheck_url ); ?>"><?php esc_html_e( 'Re-check', 'freshnews-ai-content-hub' ); ?></a>
				</p>
			</div>
			<?php
		}
	}

	/**
	 * Transient key for robots.txt sitemap check.
	 */
	const ROBOTS_TXT_CHECK_TRANSIENT = 'freshnews_robots_txt_sitemap_check';

	/**
	 * Transient key for AI discovery status (robots.txt, llms.txt, sitemap).
	 */
	const AI_DISCOVERY_STATUS_TRANSIENT = 'freshnews_ai_discovery_status';

	/**
	 * Transient key for llms.txt reachability check (with body preview).
	 */
	const LLMS_TXT_CHECK_TRANSIENT = 'freshnews_llms_txt_check';

	/**
	 * Perform robots.txt sitemap check.
	 * Prefer internal filter output (what WordPress would serve) to avoid loopback/SSL failures.
	 *
	 * @return array{ ok: bool, sitemap_line: string, fetch_error: string|null } ok is true if main Sitemap line is present.
	 */
	public function check_robots_txt_has_sitemap() {
		$signals_base = $this->get_signals_hub_base_url();
		$sitemap_url  = $signals_base . '/sitemap.xml';
		$sitemap_line = 'Sitemap: ' . $sitemap_url;

		if ( empty( $signals_base ) ) {
			return array( 'ok' => false, 'sitemap_line' => $sitemap_line, 'fetch_error' => null );
		}

		// Use the same content WordPress would serve for robots.txt (no network request).
		// This avoids "unreachable" when loopback is blocked or SSL fails (e.g. local dev).
		$virtual = apply_filters( 'robots_txt', '', '1' );
		if ( is_string( $virtual ) && stripos( $virtual, $sitemap_line ) !== false ) {
			return array( 'ok' => true, 'sitemap_line' => $sitemap_line, 'fetch_error' => null );
		}

		// Fallback: fetch live URL (e.g. when a static robots.txt might be served).
		$url = home_url( '/robots.txt' );
		$res = wp_remote_get(
			$url,
			array(
				'timeout'     => 3,
				'sslverify'   => true,
				'user-agent'  => 'FreshNews-AI-Content-Hub-Plugin/' . FRESHNEWS_AI_CONTENT_HUB_VERSION,
			)
		);

		if ( is_wp_error( $res ) ) {
			return array( 'ok' => false, 'sitemap_line' => $sitemap_line, 'fetch_error' => 'fetch_error' );
		}
		if ( wp_remote_retrieve_response_code( $res ) !== 200 ) {
			return array( 'ok' => false, 'sitemap_line' => $sitemap_line, 'fetch_error' => 'non_200' );
		}

		$body = wp_remote_retrieve_body( $res );
		$ok   = stripos( $body, $sitemap_line ) !== false;

		return array( 'ok' => $ok, 'sitemap_line' => $sitemap_line, 'fetch_error' => null );
	}

	/**
	 * Check AI discovery status: robots.txt injected, llms.txt reachable, sitemap reachable.
	 *
	 * @return array{ robots_txt: bool, llms_txt: bool, sitemap: bool }|null Null if not configured.
	 */
	public function check_ai_discovery_status() {
		$signals_base = $this->get_signals_hub_base_url();
		if ( empty( $signals_base ) ) {
			return null;
		}
		$options       = get_option( 'freshnews_ai_newsroom_options', array() );
		$robots_result = $this->check_robots_txt_has_sitemap();
		$llms_url     = home_url( '/llms.txt' );
		$llms_res     = wp_remote_get(
			$llms_url,
			array(
				'timeout'    => 5,
				'sslverify'  => true,
				'user-agent' => 'FreshNews-AI-Content-Hub-Plugin/' . FRESHNEWS_AI_CONTENT_HUB_VERSION,
			)
		);
		$llms_ok = false;
		if ( ! empty( $options['enable_llms_txt'] ) && ! is_wp_error( $llms_res ) && wp_remote_retrieve_response_code( $llms_res ) === 200 ) {
			$body = wp_remote_retrieve_body( $llms_res );
			$llms_ok = ( is_string( $body ) && trim( $body ) !== '' && strpos( $body, $signals_base ) !== false );
		}
		$sitemap_ok = false;
		if ( function_exists( 'fn_http_check' ) ) {
			$sitemap_res = fn_http_check( $signals_base . '/sitemap.xml' );
			$sitemap_ok  = $sitemap_res['ok'] && (int) $sitemap_res['status'] === 200;
		}
		return array(
			'robots_txt' => ! empty( $robots_result['ok'] ),
			'llms_txt'   => $llms_ok,
			'sitemap'    => $sitemap_ok,
		);
	}

	/**
	 * Check llms.txt reachability and return body preview for status banner.
	 *
	 * @return array{ ok: bool, body_preview: string, error: string|null }|null Null when llms disabled or no Signals URL.
	 */
	public function check_llms_txt_reachable() {
		$options       = get_option( 'freshnews_ai_newsroom_options', array() );
		$signals_base  = $this->get_signals_hub_base_url();
		if ( empty( $options['enable_llms_txt'] ) || empty( $signals_base ) ) {
			return null;
		}
		$url = home_url( '/llms.txt' );
		$res = wp_remote_get(
			$url,
			array(
				'timeout'    => 5,
				'sslverify'  => true,
				'user-agent' => 'FreshNews-AI-Content-Hub-Plugin/' . FRESHNEWS_AI_CONTENT_HUB_VERSION,
			)
		);
		if ( is_wp_error( $res ) ) {
			return array( 'ok' => false, 'body_preview' => '', 'error' => $res->get_error_message() );
		}
		$code = wp_remote_retrieve_response_code( $res );
		if ( $code !== 200 ) {
			return array( 'ok' => false, 'body_preview' => '', 'error' => 'HTTP ' . $code );
		}
		$body = wp_remote_retrieve_body( $res );
		if ( ! is_string( $body ) ) {
			return array( 'ok' => false, 'body_preview' => '', 'error' => __( 'Empty or invalid response', 'freshnews-ai-content-hub' ) );
		}
		$has_hub = strpos( $body, $signals_base ) !== false;
		$preview = mb_substr( $body, 0, 300 );
		if ( mb_strlen( $body ) > 300 ) {
			$preview .= '…';
		}
		return array(
			'ok'           => $has_hub,
			'body_preview' => $preview,
			'error'        => $has_hub ? null : __( 'Response does not contain Hub URL', 'freshnews-ai-content-hub' ),
		);
	}

	/**
	 * Render llms.txt status banner (green/red + View details with body preview). Shown only when Enable llms.txt is on.
	 */
	public function render_llms_txt_status_banner() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		if ( empty( $options['enable_llms_txt'] ) ) {
			return;
		}
		$signals_base  = $this->get_signals_hub_base_url();
		if ( empty( $signals_base ) ) {
			return;
		}
		$transient_key = self::LLMS_TXT_CHECK_TRANSIENT;
		$recheck_url   = admin_url( 'options-general.php?page=freshnews-ai-content-hub&freshnews_llms_recheck=1' );
		$recheck_url   = wp_nonce_url( $recheck_url, 'freshnews_llms_recheck', '_wpnonce' );
		if ( isset( $_GET['freshnews_llms_recheck'] ) && isset( $_GET['_wpnonce'] ) ) {
			if ( wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'freshnews_llms_recheck' ) ) {
				delete_transient( $transient_key );
			}
		}
		$cached = get_transient( $transient_key );
		if ( false !== $cached && is_array( $cached ) ) {
			$result = $cached;
		} else {
			$result = $this->check_llms_txt_reachable();
			if ( $result !== null ) {
				set_transient( $transient_key, $result, 10 * MINUTE_IN_SECONDS );
			}
		}
		if ( $result === null ) {
			return;
		}
		$ok    = ! empty( $result['ok'] );
		$preview = isset( $result['body_preview'] ) ? $result['body_preview'] : '';
		$error  = isset( $result['error'] ) ? $result['error'] : '';
		if ( $ok ) {
			?>
			<div class="notice notice-success is-dismissible freshnews-llms-status">
				<p>
					<strong><?php esc_html_e( 'llms.txt: reachable', 'freshnews-ai-content-hub' ); ?></strong>
					<a href="<?php echo esc_url( $recheck_url ); ?>"><?php esc_html_e( 'Re-check', 'freshnews-ai-content-hub' ); ?></a>
				</p>
				<?php if ( $preview !== '' ) : ?>
				<details class="freshnews-llms-details" style="margin-top:8px;">
					<summary><?php esc_html_e( 'View details', 'freshnews-ai-content-hub' ); ?></summary>
					<pre style="margin:8px 0 0 0; padding:8px; background:#f0f0f1; border:1px solid #c3c4c7; border-radius:4px; font-size:12px; white-space:pre-wrap; word-break:break-all; max-height:200px; overflow:auto;"><?php echo esc_html( $preview ); ?></pre>
				</details>
				<?php endif; ?>
			</div>
			<?php
		} else {
			?>
			<div class="notice notice-error is-dismissible freshnews-llms-status">
				<p>
					<strong><?php esc_html_e( 'llms.txt: not reachable or missing required content', 'freshnews-ai-content-hub' ); ?></strong>
				</p>
				<?php if ( $error ) : ?>
				<p><?php echo esc_html( $error ); ?></p>
				<?php endif; ?>
				<?php if ( $preview !== '' ) : ?>
				<details class="freshnews-llms-details" style="margin-top:8px;">
					<summary><?php esc_html_e( 'View details', 'freshnews-ai-content-hub' ); ?></summary>
					<pre style="margin:8px 0 0 0; padding:8px; background:#f0f0f1; border:1px solid #c3c4c7; border-radius:4px; font-size:12px; white-space:pre-wrap; word-break:break-all; max-height:200px; overflow:auto;"><?php echo esc_html( $preview ); ?></pre>
				</details>
				<?php endif; ?>
				<p><a href="<?php echo esc_url( $recheck_url ); ?>"><?php esc_html_e( 'Re-check', 'freshnews-ai-content-hub' ); ?></a></p>
			</div>
			<?php
		}
	}

	/**
	 * Render AI discovery status indicators on settings page (robots.txt, llms.txt, sitemap).
	 */
	public function render_ai_discovery_status() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		if ( empty( $options['enable_robots_sitemaps'] ) ) {
			return;
		}
		$signals_base = $this->get_signals_hub_base_url();
		if ( empty( $signals_base ) ) {
			return;
		}
		$transient_key = self::AI_DISCOVERY_STATUS_TRANSIENT;
		$recheck_url   = admin_url( 'options-general.php?page=freshnews-ai-content-hub&freshnews_ai_discovery_recheck=1' );
		$recheck_url   = wp_nonce_url( $recheck_url, 'freshnews_ai_discovery_recheck', '_wpnonce' );
		if ( isset( $_GET['freshnews_ai_discovery_recheck'] ) && isset( $_GET['_wpnonce'] ) ) {
			if ( wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'freshnews_ai_discovery_recheck' ) ) {
				delete_transient( $transient_key );
			}
		}
		$cached = get_transient( $transient_key );
		if ( false !== $cached && is_array( $cached ) ) {
			$status = $cached;
		} else {
			$status = $this->check_ai_discovery_status();
			if ( $status !== null ) {
				set_transient( $transient_key, $status, 10 * MINUTE_IN_SECONDS );
			}
		}
		if ( $status === null ) {
			return;
		}
		$items = array(
			'robots_txt' => __( 'robots.txt injected', 'freshnews-ai-content-hub' ),
			'sitemap'    => __( 'sitemap reachable', 'freshnews-ai-content-hub' ),
		);
		if ( ! empty( $options['enable_llms_txt'] ) ) {
			$items = array(
				'robots_txt' => $items['robots_txt'],
				'llms_txt'   => __( 'llms.txt reachable', 'freshnews-ai-content-hub' ),
				'sitemap'    => $items['sitemap'],
			);
		}
		?>
		<div class="freshnews-ai-discovery-status" style="margin-top:16px;margin-bottom:0;">
			<h3 style="margin:0 0 8px 0; font-size:1em;"><?php esc_html_e( 'AI Discovery', 'freshnews-ai-content-hub' ); ?></h3>
			<div class="freshnews-ai-discovery-rows" style="display:flex;flex-direction:column;gap:6px;">
				<?php foreach ( $items as $key => $label ) : ?>
					<?php
					$ok = ! empty( $status[ $key ] );
					?>
					<div class="freshnews-ai-discovery-row" style="display:flex;align-items:center;gap:8px;">
						<span class="freshnews-ai-discovery-dot" style="width:12px;height:12px;border-radius:50%;flex-shrink:0;background:<?php echo $ok ? '#00a32a' : '#d63638'; ?>;" aria-hidden="true"></span>
						<span style="font-size:13px;"><?php echo esc_html( $label ); ?></span>
						<span style="color:#646970;font-size:12px;">(<?php echo $ok ? esc_html__( 'OK', 'freshnews-ai-content-hub' ) : esc_html__( 'Not detected', 'freshnews-ai-content-hub' ); ?>)</span>
					</div>
				<?php endforeach; ?>
			</div>
			<p style="margin:8px 0 0 0;">
				<a href="<?php echo esc_url( $recheck_url ); ?>"><?php esc_html_e( 'Re-check', 'freshnews-ai-content-hub' ); ?></a>
			</p>
		</div>
		<?php
	}

	/**
	 * Render robots.txt sitemap status banner on settings page.
	 * Only shown when "Enable robots.txt sitemaps" is checked and Signals URL is configured.
	 */
	public function render_robots_txt_status_banner() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		if ( empty( $options['enable_robots_sitemaps'] ) ) {
			return;
		}
		$signals_base = $this->get_signals_hub_base_url();
		if ( empty( $signals_base ) ) {
			return;
		}

		$transient_key = self::ROBOTS_TXT_CHECK_TRANSIENT;
		$recheck_url   = admin_url( 'options-general.php?page=freshnews-ai-content-hub&freshnews_robots_recheck=1' );
		$recheck_url   = wp_nonce_url( $recheck_url, 'freshnews_robots_recheck', '_wpnonce' );
		$robots_url    = home_url( '/robots.txt' );

		// Handle recheck: clear transient
		if ( isset( $_GET['freshnews_robots_recheck'] ) && isset( $_GET['_wpnonce'] ) ) {
			if ( wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'freshnews_robots_recheck' ) ) {
				delete_transient( $transient_key );
			}
		}

		$cached = get_transient( $transient_key );
		if ( false !== $cached && is_array( $cached ) ) {
			$result = $cached;
		} else {
			$result = $this->check_robots_txt_has_sitemap();
			set_transient( $transient_key, $result, 10 * MINUTE_IN_SECONDS );
		}

		if ( $result['ok'] ) {
			?>
			<div class="notice notice-success is-dismissible">
				<p>
					<strong><?php esc_html_e( 'robots.txt is configured', 'freshnews-ai-content-hub' ); ?></strong>
					<?php esc_html_e( 'Crawlers can discover your Signals Hub sitemaps.', 'freshnews-ai-content-hub' ); ?>
					<a href="<?php echo esc_url( $recheck_url ); ?>"><?php esc_html_e( 'Re-check', 'freshnews-ai-content-hub' ); ?></a>
				</p>
			</div>
			<?php
		} else {
			$fetch_error = isset( $result['fetch_error'] ) ? $result['fetch_error'] : null;
			?>
			<div class="notice notice-warning is-dismissible">
				<p>
					<strong>
						<?php
						if ( $fetch_error ) {
							esc_html_e( 'robots.txt unreachable', 'freshnews-ai-content-hub' );
						} else {
							esc_html_e( 'robots.txt may be overridden or static', 'freshnews-ai-content-hub' );
						}
						?>
					</strong>
					<?php
					if ( $fetch_error ) {
						esc_html_e( 'WordPress could not fetch your robots.txt (network error or non-200 response).', 'freshnews-ai-content-hub' );
						echo ' ';
						esc_html_e( 'Your site may use a static robots.txt file, or crawlers might not be able to discover the Signals sitemaps via the standard URL.', 'freshnews-ai-content-hub' );
					} else {
						esc_html_e( 'The Signals Hub sitemap line was not found in your robots.txt.', 'freshnews-ai-content-hub' );
					}
					?>
				</p>
				<p>
					<?php
					if ( $fetch_error ) {
						esc_html_e( 'Likely causes: static robots.txt in site root, CDN/cache, security plugin, or robots.txt served by the server before WordPress.', 'freshnews-ai-content-hub' );
					} else {
						esc_html_e( 'Likely causes: SEO plugin override, CDN/cache, static robots.txt file in site root, or security plugin.', 'freshnews-ai-content-hub' );
					}
					?>
				</p>
				<p>
					<?php esc_html_e( 'Next step: ', 'freshnews-ai-content-hub' ); ?>
					<a href="<?php echo esc_url( $robots_url ); ?>" target="_blank" rel="noopener"><?php esc_html_e( 'Open robots.txt', 'freshnews-ai-content-hub' ); ?></a>
					<?php esc_html_e( ' and add these Sitemap lines manually if needed:', 'freshnews-ai-content-hub' ); ?>
				</p>
				<p><code><?php echo esc_html( $result['sitemap_line'] ); ?><br><?php echo esc_html( 'Sitemap: ' . $signals_base . '/news-sitemap.xml' ); ?></code></p>
				<p>
					<a href="<?php echo esc_url( $recheck_url ); ?>"><?php esc_html_e( 'Re-check', 'freshnews-ai-content-hub' ); ?></a>
				</p>
			</div>
			<?php
		}
	}

	/**
	 * Register settings
	 */
	public function register_settings() {
		register_setting(
			'freshnews_ai_newsroom_options',
			'freshnews_ai_newsroom_options',
			array( $this, 'sanitize_settings' )
		);

		// General settings section
		add_settings_section(
			'freshnews_ai_newsroom_general',
			__( 'General Settings', 'freshnews-ai-content-hub' ),
			null,
			'freshnews_ai_newsroom'
		);

		// Signals URL field
		add_settings_field(
			'newsroom_url',
			__( 'Signals URL', 'freshnews-ai-content-hub' ) . ' <span class="required">*</span>',
			array( $this, 'render_newsroom_url_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_general'
		);

		// SEO & Discovery section
		add_settings_section(
			'freshnews_ai_newsroom_seo',
			__( 'SEO & Discovery', 'freshnews-ai-content-hub' ),
			null,
			'freshnews_ai_newsroom'
		);

		// Enable robots.txt sitemaps field
		add_settings_field(
			'enable_robots_sitemaps',
			__( 'Enable robots.txt sitemaps', 'freshnews-ai-content-hub' ) . ' <span class="recommended">(recommended)</span>',
			array( $this, 'render_enable_robots_sitemaps_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_seo'
		);

		// Enable llms.txt field
		add_settings_field(
			'enable_llms_txt',
			__( 'Enable llms.txt', 'freshnews-ai-content-hub' ),
			array( $this, 'render_enable_llms_txt_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_seo'
		);

		// Footer (Latest Signal) section
		add_settings_section(
			'freshnews_ai_newsroom_footer',
			__( 'Footer (Latest Signal)', 'freshnews-ai-content-hub' ),
			null,
			'freshnews_ai_newsroom'
		);

		// Enable footer integration
		add_settings_field(
			'footer_pulse_enabled',
			__( 'Enable footer integration', 'freshnews-ai-content-hub' ),
			array( $this, 'render_footer_pulse_enabled_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_footer'
		);

		// Language mode
		add_settings_field(
			'footer_pulse_lang_mode',
			__( 'Language mode', 'freshnews-ai-content-hub' ),
			array( $this, 'render_footer_pulse_lang_mode_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_footer'
		);

		// Footer lang (when mode=fixed)
		add_settings_field(
			'footer_pulse_lang',
			__( 'Footer language (fixed mode)', 'freshnews-ai-content-hub' ),
			array( $this, 'render_footer_pulse_lang_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_footer'
		);

		// Widget settings section
		add_settings_section(
			'freshnews_ai_newsroom_widget',
			__( 'Widget Settings', 'freshnews-ai-content-hub' ),
			null,
			'freshnews_ai_newsroom'
		);

		// Widget mode field
		add_settings_field(
			'widget_mode',
			__( 'Widget Mode', 'freshnews-ai-content-hub' ),
			array( $this, 'render_widget_mode_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_widget'
		);

		// Widget kind field
		add_settings_field(
			'widget_kind',
			__( 'Article Kind', 'freshnews-ai-content-hub' ),
			array( $this, 'render_widget_kind_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_widget'
		);

		// Widget limit field
		add_settings_field(
			'widget_limit',
			__( 'Article Limit', 'freshnews-ai-content-hub' ),
			array( $this, 'render_widget_limit_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_widget'
		);

		// Link base field
		add_settings_field(
			'link_base',
			__( 'Link Base URL', 'freshnews-ai-content-hub' ) . ' <span class="optional">(optional)</span>',
			array( $this, 'render_link_base_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_widget'
		);

		// Page settings section
		add_settings_section(
			'freshnews_ai_newsroom_page',
			__( 'Page Settings', 'freshnews-ai-content-hub' ),
			null,
			'freshnews_ai_newsroom'
		);

		// Enabled field
		add_settings_field(
			'enabled',
			__( 'Enabled', 'freshnews-ai-content-hub' ),
			array( $this, 'render_enabled_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_page'
		);

		// Page slug field
		add_settings_field(
			'page_slug',
			__( 'Page Slug', 'freshnews-ai-content-hub' ),
			array( $this, 'render_page_slug_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_page'
		);

		// Page title field
		add_settings_field(
			'page_title',
			__( 'Signals Page Title', 'freshnews-ai-content-hub' ),
			array( $this, 'render_page_title_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_page'
		);

		// Enable Insights page field
		add_settings_field(
			'enable_insights_page',
			__( 'Enable Insights Page', 'freshnews-ai-content-hub' ),
			array( $this, 'render_enable_insights_page_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_page'
		);

		// Insights page slug field
		add_settings_field(
			'insights_page_slug',
			__( 'Insights Page Slug', 'freshnews-ai-content-hub' ),
			array( $this, 'render_insights_page_slug_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_page'
		);

		// Insights page title field
		add_settings_field(
			'insights_page_title',
			__( 'Insights Page Title', 'freshnews-ai-content-hub' ),
			array( $this, 'render_insights_page_title_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_page'
		);

		// Enable FAQ page field
		add_settings_field(
			'enable_faq_page',
			__( 'Enable FAQ Page', 'freshnews-ai-content-hub' ),
			array( $this, 'render_enable_faq_page_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_page'
		);

		// Auto-create page field
		add_settings_field(
			'auto_create_page',
			__( 'Auto-create or update WordPress Pages', 'freshnews-ai-content-hub' ),
			array( $this, 'render_auto_create_page_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_page'
		);

		// Advanced settings section
		add_settings_section(
			'freshnews_ai_newsroom_advanced',
			__( 'Advanced Settings', 'freshnews-ai-content-hub' ),
			null,
			'freshnews_ai_newsroom'
		);

		// Language field
		add_settings_field(
			'language',
			__( 'Language / Locale', 'freshnews-ai-content-hub' ),
			array( $this, 'render_language_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_advanced'
		);

		// Theme field
		add_settings_field(
			'theme',
			__( 'Theme', 'freshnews-ai-content-hub' ),
			array( $this, 'render_theme_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_advanced'
		);

		// Canonical mode field
		add_settings_field(
			'canonical_mode',
			__( 'Canonical Mode', 'freshnews-ai-content-hub' ),
			array( $this, 'render_canonical_mode_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_advanced'
		);

		// Defer script loading field
		add_settings_field(
			'defer_script_loading',
			__( 'Defer Script Loading', 'freshnews-ai-content-hub' ),
			array( $this, 'render_defer_script_loading_field' ),
			'freshnews_ai_newsroom',
			'freshnews_ai_newsroom_advanced'
		);
	}

	/**
	 * Sanitize settings
	 * Preserves existing settings to maintain backward compatibility
	 *
	 * @param array $input Raw input data.
	 * @return array Sanitized data.
	 */
	public function sanitize_settings( $input ) {
		// Get existing options to preserve settings not in current form
		$existing = get_option( 'freshnews_ai_newsroom_options', array() );
		$sanitized = array();

		// Enabled
		$sanitized['enabled'] = isset( $input['enabled'] ) && $input['enabled'];

		// Signals URL (stored as newsroom_url for backward compatibility)
		$raw_url = isset( $input['newsroom_url'] ) ? trim( (string) $input['newsroom_url'] ) : '';
		if ( ! empty( $raw_url ) ) {
			$raw_url = rtrim( $raw_url, '/' );
			if ( ! preg_match( '#^https?://#i', $raw_url ) ) {
				$raw_url = 'https://' . $raw_url;
			}
		}
		$sanitized['newsroom_url'] = ! empty( $raw_url ) ? esc_url_raw( $raw_url ) : '';
		if ( ! empty( $sanitized['newsroom_url'] ) ) {
			$test = esc_url_raw( $sanitized['newsroom_url'] );
			if ( empty( $test ) || ( strpos( $test, 'http://' ) !== 0 && strpos( $test, 'https://' ) !== 0 ) ) {
				add_settings_error(
					'freshnews_ai_newsroom_options',
					'invalid_signals_url',
					__( 'Signals URL is invalid. Please enter a valid URL (e.g., https://signals.yourdomain.com).', 'freshnews-ai-content-hub' ),
					'error'
				);
				$sanitized['newsroom_url'] = isset( $existing['newsroom_url'] ) ? $existing['newsroom_url'] : '';
			}
		}

		// Page slug
		$sanitized['page_slug'] = isset( $input['page_slug'] ) ? sanitize_title( trim( $input['page_slug'] ) ) : 'signals';

		// Page title
		$sanitized['page_title'] = isset( $input['page_title'] ) ? sanitize_text_field( trim( $input['page_title'] ) ) : 'Signals';

		// Language
		if ( isset( $input['language'] ) && ! empty( trim( $input['language'] ) ) ) {
			$sanitized['language'] = preg_replace( '/[^a-z-]/i', '', trim( $input['language'] ) );
			// If sanitization results in empty string, default to 'en'
			if ( empty( $sanitized['language'] ) ) {
				$sanitized['language'] = 'en';
			}
		} else {
			$sanitized['language'] = 'en';
		}

		// Theme
		$allowed_themes = array( 'light', 'dark', 'auto' );
		$sanitized['theme'] = isset( $input['theme'] ) && in_array( $input['theme'], $allowed_themes, true ) ? $input['theme'] : 'auto';

		// Canonical mode
		$allowed_canonical_modes = array( 'use_newsroom_url', 'disable' );
		$sanitized['canonical_mode'] = isset( $input['canonical_mode'] ) && in_array( $input['canonical_mode'], $allowed_canonical_modes, true ) ? $input['canonical_mode'] : 'use_newsroom_url';

		// Auto-create page
		$sanitized['auto_create_page'] = isset( $input['auto_create_page'] ) && $input['auto_create_page'];

		// Defer script loading
		$sanitized['defer_script_loading'] = isset( $input['defer_script_loading'] ) && $input['defer_script_loading'];

		// Enable robots.txt sitemaps
		$sanitized['enable_robots_sitemaps'] = isset( $input['enable_robots_sitemaps'] ) && $input['enable_robots_sitemaps'];

		// Enable llms.txt
		$sanitized['enable_llms_txt'] = isset( $input['enable_llms_txt'] ) && $input['enable_llms_txt'];

		// Footer pulse (Latest Signal)
		$sanitized['footer_pulse_enabled']   = isset( $input['footer_pulse_enabled'] ) && $input['footer_pulse_enabled'];
		$allowed_lang_modes                  = array( 'auto', 'fixed' );
		$sanitized['footer_pulse_lang_mode'] = isset( $input['footer_pulse_lang_mode'] ) && in_array( $input['footer_pulse_lang_mode'], $allowed_lang_modes, true ) ? $input['footer_pulse_lang_mode'] : 'auto';
		$sanitized['footer_pulse_lang']      = isset( $input['footer_pulse_lang'] ) ? sanitize_text_field( trim( $input['footer_pulse_lang'] ) ) : ( isset( $existing['footer_pulse_lang'] ) ? $existing['footer_pulse_lang'] : 'en' );

		// Widget mode
		$allowed_modes = array( 'insights-list', 'signals-list', 'article' );
		$sanitized['widget_mode'] = isset( $input['widget_mode'] ) && in_array( $input['widget_mode'], $allowed_modes, true ) ? $input['widget_mode'] : 'signals-list';

		// Widget kind
		$allowed_kinds = array( 'daily', 'insight', 'how_to', 'analysis' );
		$sanitized['widget_kind'] = isset( $input['widget_kind'] ) && in_array( $input['widget_kind'], $allowed_kinds, true ) ? $input['widget_kind'] : 'daily';

		// Widget limit
		$sanitized['widget_limit'] = isset( $input['widget_limit'] ) ? absint( $input['widget_limit'] ) : 20;
		if ( $sanitized['widget_limit'] > 100 ) {
			$sanitized['widget_limit'] = 100;
		}
		if ( $sanitized['widget_limit'] < 1 ) {
			$sanitized['widget_limit'] = 20;
		}

		// Link base
		$sanitized['link_base'] = isset( $input['link_base'] ) ? esc_url_raw( trim( $input['link_base'] ) ) : '';

		// Enable Insights page
		$sanitized['enable_insights_page'] = isset( $input['enable_insights_page'] ) && $input['enable_insights_page'];

		// Enable FAQ page
		$sanitized['enable_faq_page'] = isset( $input['enable_faq_page'] ) && $input['enable_faq_page'];

		// Insights page slug and title (only if enabled)
		if ( $sanitized['enable_insights_page'] ) {
			$sanitized['insights_page_slug']  = isset( $input['insights_page_slug'] ) ? sanitize_title( trim( $input['insights_page_slug'] ) ) : 'insights';
			$sanitized['insights_page_title'] = isset( $input['insights_page_title'] ) ? sanitize_text_field( trim( $input['insights_page_title'] ) ) : 'Insights';
		} else {
			// Clear insights page settings if disabled
			$sanitized['insights_page_slug']  = '';
			$sanitized['insights_page_title']  = '';
		}

		// Preserve any existing settings that aren't in the current form
		// This ensures backward compatibility when adding new features
		$deprecated_keys = array( 'footer_pulse_label' );
		foreach ( $existing as $key => $value ) {
			if ( ! isset( $sanitized[ $key ] ) && ! in_array( $key, $deprecated_keys, true ) ) {
				$sanitized[ $key ] = $value;
			}
		}

		return $sanitized;
	}

	/**
	 * Add admin menu
	 */
	public function add_admin_menu() {
		add_options_page(
			__( 'FreshNews AI Content Hub', 'freshnews-ai-content-hub' ),
			__( 'FreshNews AI Content Hub', 'freshnews-ai-content-hub' ),
			'manage_options',
			'freshnews-ai-content-hub',
			array( $this, 'render_settings_page' )
		);
	}

	/**
	 * Render settings page
	 */
	public function render_settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$this->render_robots_txt_status_banner();
		$this->render_ai_discovery_status();
		$this->render_llms_txt_status_banner();
		$this->render_footer_pulse_status_banner();
		include FRESHNEWS_AI_CONTENT_HUB_PLUGIN_DIR . 'includes/admin-settings.php';
		$this->render_health_checks_section();
	}

	/**
	 * Render enabled field
	 */
	public function render_enabled_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['enabled'] ) ? $options['enabled'] : true;
		?>
		<input type="checkbox" id="enabled" name="freshnews_ai_newsroom_options[enabled]" value="1" <?php checked( $value, true ); ?> />
		<label for="enabled"><?php esc_html_e( 'Enable FreshNews.ai signals embedding', 'freshnews-ai-content-hub' ); ?></label>
		<?php
	}

	/**
	 * Render signals URL field
	 */
	public function render_newsroom_url_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['newsroom_url'] ) ? esc_attr( $options['newsroom_url'] ) : '';
		$embed_script = function_exists( 'freshnews_get_embed_script_url' ) ? freshnews_get_embed_script_url() : '';
		$embed_css    = function_exists( 'freshnews_get_embed_css_url' ) ? freshnews_get_embed_css_url() : '';
		?>
		<input type="url" id="newsroom_url" name="freshnews_ai_newsroom_options[newsroom_url]" value="<?php echo esc_attr( $value ); ?>" class="regular-text" placeholder="https://signals.yourdomain.com" />
		<p class="description"><?php esc_html_e( 'Example: https://signals.yourdomain.com (your CNAME should point to ingest.freshnews.ai)', 'freshnews-ai-content-hub' ); ?></p>
		<?php if ( ! empty( $embed_script ) ) : ?>
		<details class="freshnews-computed-urls" style="margin-top:8px;">
			<summary><?php esc_html_e( 'Computed URLs (read-only)', 'freshnews-ai-content-hub' ); ?></summary>
			<p class="description" style="margin:6px 0 0 0;">
				<strong><?php esc_html_e( 'Embed script:', 'freshnews-ai-content-hub' ); ?></strong> <code><?php echo esc_html( $embed_script ); ?></code><br>
				<strong><?php esc_html_e( 'Embed CSS:', 'freshnews-ai-content-hub' ); ?></strong> <code><?php echo esc_html( $embed_css ); ?></code>
			</p>
		</details>
		<?php endif; ?>
		<?php
	}

	/**
	 * Render enable robots.txt sitemaps field
	 */
	public function render_enable_robots_sitemaps_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['enable_robots_sitemaps'] ) ? $options['enable_robots_sitemaps'] : true;
		$signals_base = $this->get_signals_hub_base_url();
		$example_url  = $signals_base ? $signals_base . '/sitemap.xml' : 'https://signals.yourdomain.com/sitemap.xml';
		?>
		<input type="checkbox" id="enable_robots_sitemaps" name="freshnews_ai_newsroom_options[enable_robots_sitemaps]" value="1" <?php checked( $value, true ); ?> />
		<label for="enable_robots_sitemaps"><?php esc_html_e( 'Add Signals Hub sitemaps to your site\'s robots.txt', 'freshnews-ai-content-hub' ); ?></label>
		<p class="description"><?php esc_html_e( 'Adds: Sitemap: URL (and Google News sitemap if available). Helps crawlers discover your Signals content.', 'freshnews-ai-content-hub' ); ?></p>
		<p class="description"><code>Sitemap: <?php echo esc_html( $example_url ); ?></code></p>
		<?php
	}

	/**
	 * Render enable llms.txt field
	 */
	public function render_enable_llms_txt_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['enable_llms_txt'] ) ? $options['enable_llms_txt'] : false;
		?>
		<input type="checkbox" id="enable_llms_txt" name="freshnews_ai_newsroom_options[enable_llms_txt]" value="1" <?php checked( $value, true ); ?> />
		<label for="enable_llms_txt"><?php esc_html_e( 'Enable llms.txt', 'freshnews-ai-content-hub' ); ?></label>
		<p class="description"><?php esc_html_e( 'Serves /llms.txt to help AI crawlers discover your Signals Hub content.', 'freshnews-ai-content-hub' ); ?></p>
		<?php
	}

	/**
	 * Render widget mode field
	 */
	public function render_widget_mode_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['widget_mode'] ) ? $options['widget_mode'] : 'signals-list';
		?>
		<select id="widget_mode" name="freshnews_ai_newsroom_options[widget_mode]">
			<option value="signals-list" <?php selected( $value, 'signals-list' ); ?>><?php esc_html_e( 'Signals List', 'freshnews-ai-content-hub' ); ?></option>
			<option value="insights-list" <?php selected( $value, 'insights-list' ); ?>><?php esc_html_e( 'Insights List', 'freshnews-ai-content-hub' ); ?></option>
			<option value="article" <?php selected( $value, 'article' ); ?>><?php esc_html_e( 'Single Article', 'freshnews-ai-content-hub' ); ?></option>
		</select>
		<p class="description"><?php esc_html_e( 'Display mode for the widget', 'freshnews-ai-content-hub' ); ?></p>
		<?php
	}

	/**
	 * Render widget kind field
	 */
	public function render_widget_kind_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['widget_kind'] ) ? $options['widget_kind'] : 'daily';
		?>
		<select id="widget_kind" name="freshnews_ai_newsroom_options[widget_kind]">
			<option value="daily" <?php selected( $value, 'daily' ); ?>><?php esc_html_e( 'Daily', 'freshnews-ai-content-hub' ); ?></option>
			<option value="insight" <?php selected( $value, 'insight' ); ?>><?php esc_html_e( 'Insight', 'freshnews-ai-content-hub' ); ?></option>
			<option value="how_to" <?php selected( $value, 'how_to' ); ?>><?php esc_html_e( 'How To', 'freshnews-ai-content-hub' ); ?></option>
			<option value="analysis" <?php selected( $value, 'analysis' ); ?>><?php esc_html_e( 'Analysis', 'freshnews-ai-content-hub' ); ?></option>
		</select>
		<p class="description"><?php esc_html_e( 'Filter articles by kind (only applies to list modes)', 'freshnews-ai-content-hub' ); ?></p>
		<?php
	}

	/**
	 * Render widget limit field
	 */
	public function render_widget_limit_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['widget_limit'] ) ? absint( $options['widget_limit'] ) : 20;
		?>
		<input type="number" id="widget_limit" name="freshnews_ai_newsroom_options[widget_limit]" value="<?php echo esc_attr( $value ); ?>" min="1" max="100" />
		<p class="description"><?php esc_html_e( 'Maximum number of articles to display (1-100, only applies to list modes)', 'freshnews-ai-content-hub' ); ?></p>
		<?php
	}

	/**
	 * Render link base field
	 */
	public function render_link_base_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['link_base'] ) ? esc_attr( $options['link_base'] ) : '';
		?>
		<input type="url" id="link_base" name="freshnews_ai_newsroom_options[link_base]" value="<?php echo esc_attr( $value ); ?>" class="regular-text" placeholder="https://your-site.com/signals" />
		<p class="description">
			<?php esc_html_e( 'Optional: Base URL for article links. If empty, uses signals URL format.', 'freshnews-ai-content-hub' ); ?>
			<br>
			<strong><?php esc_html_e( 'Note:', 'freshnews-ai-content-hub' ); ?></strong>
			<?php esc_html_e( 'For in-place article navigation (no page reload), leave this empty or set it to the same domain as your Signals URL. If set to a different domain, article links will navigate normally (full page load).', 'freshnews-ai-content-hub' ); ?>
		</p>
		<?php
	}

	/**
	 * Render page slug field
	 */
	public function render_page_slug_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['page_slug'] ) ? esc_attr( $options['page_slug'] ) : 'signals';
		?>
		<input type="text" id="page_slug" name="freshnews_ai_newsroom_options[page_slug]" value="<?php echo esc_attr( $value ); ?>" class="regular-text" />
		<p class="description"><?php esc_html_e( 'Slug for the auto-created Signals page (e.g., "signals") - displays daily news articles', 'freshnews-ai-content-hub' ); ?></p>
		<?php
	}

	/**
	 * Render page title field
	 */
	public function render_page_title_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['page_title'] ) ? esc_attr( $options['page_title'] ) : 'Signals';
		?>
		<input type="text" id="page_title" name="freshnews_ai_newsroom_options[page_title]" value="<?php echo esc_attr( $value ); ?>" class="regular-text" />
		<p class="description"><?php esc_html_e( 'Title for the auto-created Signals page (displays daily news articles)', 'freshnews-ai-content-hub' ); ?></p>
		<?php
	}

	/**
	 * Render enable insights page field
	 */
	public function render_enable_insights_page_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['enable_insights_page'] ) ? $options['enable_insights_page'] : false;
		?>
		<input type="checkbox" id="enable_insights_page" name="freshnews_ai_newsroom_options[enable_insights_page]" value="1" <?php checked( $value, true ); ?> />
		<label for="enable_insights_page"><?php esc_html_e( 'Enable separate Insights page (for users with insights content)', 'freshnews-ai-content-hub' ); ?></label>
		<p class="description"><?php esc_html_e( 'If unchecked, only the Signals page will be created. Check this if you have insights articles to display.', 'freshnews-ai-content-hub' ); ?></p>
		<?php
	}

	/**
	 * Render insights page slug field
	 */
	public function render_insights_page_slug_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$enabled = isset( $options['enable_insights_page'] ) && $options['enable_insights_page'];
		$value   = isset( $options['insights_page_slug'] ) ? esc_attr( $options['insights_page_slug'] ) : 'insights';
		?>
		<input type="text" id="insights_page_slug" name="freshnews_ai_newsroom_options[insights_page_slug]" value="<?php echo esc_attr( $value ); ?>" class="regular-text" <?php echo $enabled ? '' : 'disabled'; ?> />
		<p class="description"><?php esc_html_e( 'Slug for the auto-created Insights page (e.g., "insights")', 'freshnews-ai-content-hub' ); ?></p>
		<?php
	}

	/**
	 * Render insights page title field
	 */
	public function render_insights_page_title_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$enabled = isset( $options['enable_insights_page'] ) && $options['enable_insights_page'];
		$value   = isset( $options['insights_page_title'] ) ? esc_attr( $options['insights_page_title'] ) : 'Insights';
		?>
		<input type="text" id="insights_page_title" name="freshnews_ai_newsroom_options[insights_page_title]" value="<?php echo esc_attr( $value ); ?>" class="regular-text" <?php echo $enabled ? '' : 'disabled'; ?> />
		<p class="description"><?php esc_html_e( 'Title for the auto-created Insights page (displays insights articles)', 'freshnews-ai-content-hub' ); ?></p>
		<?php
	}

	/**
	 * Render enable FAQ page field
	 */
	public function render_enable_faq_page_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['enable_faq_page'] ) ? $options['enable_faq_page'] : false;
		?>
		<input type="checkbox" id="enable_faq_page" name="freshnews_ai_newsroom_options[enable_faq_page]" value="1" <?php checked( $value, true ); ?> />
		<label for="enable_faq_page"><?php esc_html_e( 'Enable FAQ Page', 'freshnews-ai-content-hub' ); ?></label>
		<p class="description"><?php esc_html_e( 'Create a dedicated FAQ page that renders the FreshNews.ai FAQ widget.', 'freshnews-ai-content-hub' ); ?></p>
		<?php
	}

	/**
	 * Render auto-create page field
	 */
	public function render_auto_create_page_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['auto_create_page'] ) ? $options['auto_create_page'] : true;
		?>
		<input type="checkbox" id="auto_create_page" name="freshnews_ai_newsroom_options[auto_create_page]" value="1" <?php checked( $value, true ); ?> />
		<label for="auto_create_page"><?php esc_html_e( 'Automatically create or update a WordPress page with the shortcode', 'freshnews-ai-content-hub' ); ?></label>
		<?php
	}

	/**
	 * Render language field
	 */
	public function render_language_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['language'] ) && ! empty( $options['language'] ) ? esc_attr( $options['language'] ) : 'en';
		?>
		<input type="text" id="language" name="freshnews_ai_newsroom_options[language]" value="<?php echo esc_attr( $value ); ?>" class="small-text" placeholder="en" />
		<p class="description"><?php esc_html_e( 'Language code (e.g., "en", "en-US"). Default: "en"', 'freshnews-ai-content-hub' ); ?></p>
		<?php
	}

	/**
	 * Render theme field
	 */
	public function render_theme_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['theme'] ) ? $options['theme'] : 'auto';
		?>
		<select id="theme" name="freshnews_ai_newsroom_options[theme]">
			<option value="auto" <?php selected( $value, 'auto' ); ?>><?php esc_html_e( 'Auto', 'freshnews-ai-content-hub' ); ?></option>
			<option value="light" <?php selected( $value, 'light' ); ?>><?php esc_html_e( 'Light', 'freshnews-ai-content-hub' ); ?></option>
			<option value="dark" <?php selected( $value, 'dark' ); ?>><?php esc_html_e( 'Dark', 'freshnews-ai-content-hub' ); ?></option>
		</select>
		<p class="description"><?php esc_html_e( 'Theme preference for the widget', 'freshnews-ai-content-hub' ); ?></p>
		<?php
	}

	/**
	 * Render canonical mode field
	 */
	public function render_canonical_mode_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['canonical_mode'] ) ? $options['canonical_mode'] : 'use_newsroom_url';
		?>
		<select id="canonical_mode" name="freshnews_ai_newsroom_options[canonical_mode]">
			<option value="use_newsroom_url" <?php selected( $value, 'use_newsroom_url' ); ?>><?php esc_html_e( 'Use Primary Content URL', 'freshnews-ai-content-hub' ); ?></option>
			<option value="disable" <?php selected( $value, 'disable' ); ?>><?php esc_html_e( 'Disable', 'freshnews-ai-content-hub' ); ?></option>
		</select>
		<p class="description"><?php esc_html_e( 'Canonical link handling for E-E-A-T SEO', 'freshnews-ai-content-hub' ); ?></p>
		<?php
	}

	/**
	 * Render defer script loading field
	 */
	public function render_defer_script_loading_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['defer_script_loading'] ) ? $options['defer_script_loading'] : true;
		?>
		<input type="checkbox" id="defer_script_loading" name="freshnews_ai_newsroom_options[defer_script_loading]" value="1" <?php checked( $value, true ); ?> />
		<label for="defer_script_loading"><?php esc_html_e( 'Defer script loading for better performance', 'freshnews-ai-content-hub' ); ?></label>
		<?php
	}

	/**
	 * Render footer pulse enabled field
	 */
	public function render_footer_pulse_enabled_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = ! empty( $options['footer_pulse_enabled'] ) || ! empty( $options['auto_footer_latest'] );
		?>
		<input type="checkbox" id="footer_pulse_enabled" name="freshnews_ai_newsroom_options[footer_pulse_enabled]" value="1" <?php checked( $value, true ); ?> />
		<label for="footer_pulse_enabled"><?php esc_html_e( 'Inject hidden Latest Signal link for SEO/crawlers', 'freshnews-ai-content-hub' ); ?></label>
		<p class="description"><?php esc_html_e( 'Adds an invisible link in the page footer that crawlers can read. Does not change your site layout.', 'freshnews-ai-content-hub' ); ?></p>
		<?php
	}

	/**
	 * Render footer pulse lang mode field
	 */
	public function render_footer_pulse_lang_mode_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['footer_pulse_lang_mode'] ) ? $options['footer_pulse_lang_mode'] : 'auto';
		?>
		<select id="footer_pulse_lang_mode" name="freshnews_ai_newsroom_options[footer_pulse_lang_mode]">
			<option value="auto" <?php selected( $value, 'auto' ); ?>><?php esc_html_e( 'Auto (detect from page)', 'freshnews-ai-content-hub' ); ?></option>
			<option value="fixed" <?php selected( $value, 'fixed' ); ?>><?php esc_html_e( 'Fixed', 'freshnews-ai-content-hub' ); ?></option>
		</select>
		<p class="description"><?php esc_html_e( 'Language for the latest signal link. Auto uses WPML/Polylang or your site language.', 'freshnews-ai-content-hub' ); ?></p>
		<?php
	}

	/**
	 * Render footer pulse lang field (when mode=fixed)
	 */
	public function render_footer_pulse_lang_field() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$value   = isset( $options['footer_pulse_lang'] ) ? $options['footer_pulse_lang'] : 'en';
		$mode    = isset( $options['footer_pulse_lang_mode'] ) ? $options['footer_pulse_lang_mode'] : 'auto';
		?>
		<input type="text" id="footer_pulse_lang" name="freshnews_ai_newsroom_options[footer_pulse_lang]" value="<?php echo esc_attr( $value ); ?>" class="small-text" placeholder="en" <?php disabled( $mode, 'auto' ); ?> />
		<p class="description"><?php esc_html_e( 'Used when Language mode is "Fixed". Examples: en, he, en-US.', 'freshnews-ai-content-hub' ); ?></p>
		<?php
	}

	/**
	 * Shortcode handler
	 *
	 * @param array  $atts Shortcode attributes.
	 * @param string $content Shortcode content.
	 * @return string Shortcode output.
	 */
	public function shortcode_handler( $atts, $content = '' ) {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );

		// Check if enabled
		if ( ! isset( $options['enabled'] ) || ! $options['enabled'] ) {
			if ( current_user_can( 'manage_options' ) ) {
				return '<p class="freshnews-ai-admin-notice">' . esc_html__( 'FreshNews.ai content hub is disabled. Enable it in Settings → FreshNews AI Content Hub', 'freshnews-ai-content-hub' ) . '</p>';
			}
			return '';
		}

		// Check required settings
		$signals_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
		if ( empty( $signals_url ) ) {
			if ( current_user_can( 'manage_options' ) ) {
				return '<p class="freshnews-ai-admin-notice">' . esc_html__( 'FreshNews.ai content hub is not configured. Please configure it in Settings → FreshNews AI Content Hub', 'freshnews-ai-content-hub' ) . '</p>';
			}
			return '';
		}

		// Set flag to enqueue scripts
		$this->shortcode_present = true;

		// Parse shortcode attributes (allow override of settings)
		// Auto-detect mode based on current page if mode not specified
		$auto_mode = null;
		if ( is_page() ) {
			$current_page = get_queried_object();
			if ( $current_page ) {
				$page_slug = $current_page->post_name;
				$options_page_slug = ! empty( $options['page_slug'] ) ? $options['page_slug'] : 'signals';
				$insights_enabled = ! empty( $options['enable_insights_page'] );
				$options_insights_slug = $insights_enabled && ! empty( $options['insights_page_slug'] ) ? $options['insights_page_slug'] : 'insights';

				if ( $insights_enabled && $page_slug === $options_insights_slug ) {
					$auto_mode = 'insights-list';
				} elseif ( $page_slug === $options_page_slug ) {
					$auto_mode = 'signals-list';
				}
			}
		}

		// Determine mode first (for kind default)
		$mode = $auto_mode ? $auto_mode : ( isset( $atts['mode'] ) ? $atts['mode'] : ( isset( $options['widget_mode'] ) ? $options['widget_mode'] : 'signals-list' ) );

		// Default values maintain backward compatibility
		// For insights-list mode, don't set kind (we want all non-daily articles)
		// For signals-list mode, default to 'daily'
		$default_kind = ( $mode === 'insights-list' ) ? '' : ( isset( $options['widget_kind'] ) ? $options['widget_kind'] : 'daily' );

		$default_atts = array(
			'mode'      => $mode,
			'kind'      => $default_kind,
			'limit'     => isset( $options['widget_limit'] ) ? absint( $options['widget_limit'] ) : 20,
			'language'  => ! empty( $options['language'] ) ? $options['language'] : 'en',
			'theme'     => isset( $options['theme'] ) ? $options['theme'] : 'auto',
			'link_base' => isset( $options['link_base'] ) ? $options['link_base'] : '',
			'slug'      => '', // For article mode
		);

		// Handle deprecated attribute names for backward compatibility
		if ( isset( $atts['locale'] ) && ! isset( $atts['language'] ) ) {
			$atts['language'] = $atts['locale']; // 'locale' deprecated, use 'language'
		}

		$atts = shortcode_atts( $default_atts, $atts, 'freshnews_newsroom' );

		// Ensure language is never empty (default to 'en')
		if ( empty( $atts['language'] ) ) {
			$atts['language'] = 'en';
		}

		// Build data attributes
		$data_attrs = array(
			'data-freshnews-widget' => 'insights',
			'data-mode'             => sanitize_text_field( $atts['mode'] ),
			'data-api-base'         => esc_url( $signals_url ),
			'data-language'         => sanitize_text_field( $atts['language'] ), // Always include language
			'data-theme'            => sanitize_text_field( $atts['theme'] ), // Include theme for about page
		);

		// Get current WordPress page path for navigation
		// This will be used for pushState to keep URLs on the WordPress site
		$current_page_path = '';

		// Try to get the page path from WordPress
		if ( is_page() ) {
			// For pages, get the page permalink
			$permalink = get_permalink();
			$parsed = wp_parse_url( $permalink );
			if ( isset( $parsed['path'] ) ) {
				$current_page_path = rtrim( $parsed['path'], '/' );
			}
		} elseif ( is_singular() ) {
			// For posts, get the permalink
			$permalink = get_permalink();
			$parsed = wp_parse_url( $permalink );
			if ( isset( $parsed['path'] ) ) {
				$current_page_path = rtrim( $parsed['path'], '/' );
			}
		}

		// Fallback: use REQUEST_URI
		if ( empty( $current_page_path ) ) {
			$request_uri = isset( $_SERVER['REQUEST_URI'] ) ? $_SERVER['REQUEST_URI'] : '';
			// Remove query string
			$request_uri = strtok( $request_uri, '?' );
			// Remove article slug if present (e.g., /signals/article-slug -> /signals)
			$request_uri = trim( $request_uri, '/' );
			$parts = explode( '/', $request_uri );
			if ( count( $parts ) >= 2 ) {
				// Likely an article URL, use the base path
				$current_page_path = '/' . $parts[0];
			} else {
				$current_page_path = '/' . $request_uri;
			}
		}

		// If path is empty or just '/', use '/signals' as default
		if ( empty( $current_page_path ) || $current_page_path === '/' ) {
			$current_page_path = '/signals';
		}

		// Set page-base for pushState navigation (keeps URLs on WordPress site)
		// This takes priority over link-base for navigation paths
		// The embed script will use this path when doing pushState, keeping URLs on WordPress domain
		if ( ! empty( $current_page_path ) ) {
			$data_attrs['data-page-base'] = $current_page_path;
		}

		// Always ensure language is set and not empty
		// Double-check to prevent null in URLs
		if ( empty( $data_attrs['data-language'] ) || $data_attrs['data-language'] === 'null' ) {
			$data_attrs['data-language'] = 'en';
		}

		// Check if we need embed navigation wrapper (define before using)
		// Only wrap if we're on a different domain than api-base (WordPress embed context)
		$needs_wrapper = false;
		if ( ! empty( $signals_url ) ) {
			$newsroom_parsed = wp_parse_url( $signals_url );
			$current_host = isset( $_SERVER['HTTP_HOST'] ) ? $_SERVER['HTTP_HOST'] : '';

			// Remove port from current host for comparison
			$current_host_clean = preg_replace( '/:\d+$/', '', $current_host );
			$newsroom_host = isset( $newsroom_parsed['host'] ) ? $newsroom_parsed['host'] : '';

			if ( ! empty( $newsroom_host ) && $current_host_clean !== $newsroom_host ) {
				// Different domains - we're in embed context, need wrapper for embed navigation
				$needs_wrapper = true;
			}
		}

		// Always add wrapper if page-base is set (indicates WordPress embed)
		if ( ! empty( $current_page_path ) && $current_page_path !== '/' ) {
			$needs_wrapper = true;
		}

		// Add data-embed-nav to container if we're in WordPress embed context
		// This ensures the embed script can find it (closest() searches up the tree)
		if ( $needs_wrapper ) {
			$data_attrs['data-embed-nav'] = 'true';
		}

		// For link-base: Don't set it if it points to a different domain than api-base
		// The embed script only intercepts clicks when link hostname matches api-base hostname
		// If link-base is on WordPress domain, links won't be intercepted and we'll have CORS issues
		// Solution: Let links point to newsroom domain (for interception), but use page-base for pushState (keeps URL on WordPress)
		if ( ! empty( $atts['link_base'] ) ) {
			$link_base = $atts['link_base'];
			$newsroom_url = $signals_url;

			// Only set link-base if it's on the same domain as api-base
			// This ensures the embed script can intercept clicks and handle fetching (no CORS issues)
			if ( preg_match( '/^https?:\/\//', $link_base ) ) {
				$link_parsed = wp_parse_url( $link_base );
				$newsroom_parsed = wp_parse_url( $newsroom_url );

				if ( isset( $link_parsed['host'] ) && isset( $newsroom_parsed['host'] )
					&& $link_parsed['host'] === $newsroom_parsed['host'] ) {
					// Same domain - safe to use (embed script will intercept)
					$data_attrs['data-link-base'] = esc_url( $link_base );
				}
				// If different domain, don't set link-base
				// Links will point to newsroom domain, embed script intercepts, page-base keeps URL on WordPress
			} else {
				// Relative path - don't set it (embed script needs absolute URLs for interception)
			}
		}
		// If link-base is not set, embed uses newsroom format which will be intercepted

		if ( 'article' !== $atts['mode'] ) {
			// List modes - add kind and limit
			// Only set kind attribute if it's not empty (insights-list should not have kind='daily')
			if ( ! empty( $atts['kind'] ) ) {
				$data_attrs['data-kind'] = sanitize_text_field( $atts['kind'] );
			}
			$data_attrs['data-limit'] = absint( $atts['limit'] );
		} else {
			// Article mode
			if ( ! empty( $atts['slug'] ) ) {
				$data_attrs['data-slug'] = sanitize_text_field( $atts['slug'] );
			}
		}

		// Build attributes string
		$attr_string = '';
		foreach ( $data_attrs as $key => $value ) {
			$attr_string .= ' ' . esc_attr( $key ) . '="' . esc_attr( $value ) . '"';
		}

		// $needs_wrapper is already defined above

		// Wrap container in a div with data-embed-nav when needed
		// The embed script uses closest() which searches up the tree
		// This ensures embed navigation works in WordPress while not breaking regular embeds
		if ( $needs_wrapper ) {
			$embed_nav_attr = ' data-embed-nav="true"';
			// Return wrapped container div
			// The wrapper ensures closest() can find data-embed-nav
			return '<div' . $embed_nav_attr . '><div id="freshnews-ai-root"' . $attr_string . '></div></div>';
		} else {
			// Regular embed context (same domain) - no wrapper needed
			// But if data-embed-nav is set on container, keep it for explicit enable
			// Return container div directly (standard embed behavior)
			return '<div id="freshnews-ai-root"' . $attr_string . '></div>';
		}
	}

	/**
	 * FAQ shortcode handler. Renders FAQ via the same script-based embed as Signals (insights.js)
	 * so the background and style match the WordPress theme.
	 *
	 * @param array  $atts Shortcode attributes (unused for now).
	 * @param string $content Shortcode content.
	 * @return string HTML output.
	 */
	public function shortcode_faq_handler( $atts, $content = '' ) {
		$this->faq_shortcode_present = true;
		$signals_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
		if ( empty( $signals_url ) ) {
			if ( current_user_can( 'manage_options' ) ) {
				return '<p class="freshnews-ai-admin-notice">' . esc_html__( 'FreshNews.ai is not configured. Set the Signals URL in Settings → FreshNews AI Content Hub.', 'freshnews-ai-content-hub' ) . '</p>';
			}
			return '';
		}
		$lang   = $this->get_llms_txt_lang();
		$base   = rtrim( $signals_url, '/' );
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$theme  = isset( $options['theme'] ) ? $options['theme'] : 'auto';

		$data_attrs = array(
			'id'                     => 'freshnews-faq-root',
			'data-freshnews-widget'  => 'insights',
			'data-mode'               => 'faq-list',
			'data-api-base'           => $base,
			'data-language'           => $lang,
			'data-theme'              => $theme,
			'data-page-base'          => '/faq',
		);

		// Optional: tenant for localhost/testing (same as main shortcode).
		$tenant = isset( $options['tenant_slug'] ) ? sanitize_text_field( $options['tenant_slug'] ) : '';
		if ( $tenant !== '' ) {
			$data_attrs['data-tenant'] = $tenant;
		}

		// Embed nav wrapper so in-embed navigation (Explore more → Signals/Insights/About) works.
		$attr_string = '';
		foreach ( $data_attrs as $key => $value ) {
			$attr_string .= ' ' . esc_attr( $key ) . '="' . esc_attr( $value ) . '"';
		}
		return '<div data-embed-nav="true"><div class="freshnews-faq-widget"' . $attr_string . '></div></div>';
	}

	/**
	 * Enqueue scripts
	 */
	public function enqueue_scripts() {
		// Enqueue when main shortcode or FAQ shortcode is present
		if ( ! $this->shortcode_present && ! $this->faq_shortcode_present ) {
			return;
		}

		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$signals_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
		$embed_script_url = function_exists( 'freshnews_get_embed_script_url' ) ? freshnews_get_embed_script_url() : '';

		// Enqueue when main shortcode (and embedding enabled) or FAQ shortcode is present.
		$main_embed_enabled = $this->shortcode_present && ! empty( $options['enabled'] ) && ! empty( $embed_script_url );
		$faq_embed_enabled  = $this->faq_shortcode_present && ! empty( $embed_script_url );
		if ( ! $main_embed_enabled && ! $faq_embed_enabled ) {
			return;
		}

		// Enqueue CSS (derived from Signals URL)
		$embed_css_url = function_exists( 'freshnews_get_embed_css_url' ) ? freshnews_get_embed_css_url() : '';
		if ( ! empty( $embed_css_url ) ) {
			wp_enqueue_style(
				'freshnews-ai-embed-css',
				$embed_css_url,
				array(),
				FRESHNEWS_AI_CONTENT_HUB_VERSION
			);
		}

		// Add inline script to hide page title and reduce whitespace (fallback for themes that don't respect CSS)
		$hide_title_script = "
		(function() {
			function hideTitleAndSpacing() {
				if (document.querySelector('[data-freshnews-widget=\"insights\"]')) {
					// Hide page titles - target common WordPress theme selectors
					var titleSelectors = [
						'h1.entry-title', 'h1.page-title', '.entry-title', '.page-title',
						'.entry-header h1', '.page-header h1', '.wp-block-post-title',
						'main h1:first-of-type', 'article h1:first-of-type',
						'.site-content h1:first-of-type', '.content-area h1:first-of-type',
						'#content h1:first-of-type', '.site-main h1:first-of-type'
					];
					titleSelectors.forEach(function(selector) {
						var titles = document.querySelectorAll(selector);
						titles.forEach(function(title) {
							// Check if it's in entry-header/page-header or is the first h1
							if (title.closest('.entry-header') || title.closest('.page-header') ||
							    (title.tagName === 'H1' && title === title.parentElement.querySelector('h1:first-of-type'))) {
								title.style.display = 'none';
								title.style.margin = '0';
								title.style.padding = '0';
								title.style.height = '0';
								title.style.overflow = 'hidden';
								title.style.lineHeight = '0';
							}
						});
					});

					// Remove top spacing from entry-header
					var entryHeaders = document.querySelectorAll('.entry-header, .page-header');
					entryHeaders.forEach(function(header) {
						header.style.marginTop = '0';
						header.style.paddingTop = '0';
					});

					// Remove top spacing from content areas that contain the embed
					var contentAreas = document.querySelectorAll('.entry-content, .post-content, .content, .site-main, main, article, .wp-block-post-content');
					contentAreas.forEach(function(area) {
						if (area.querySelector('[data-freshnews-widget=\"insights\"]')) {
							area.style.marginTop = '0';
							area.style.paddingTop = '0';
						}
					});

					// Remove top spacing from embed container and wrapper
					var embedContainer = document.querySelector('[data-freshnews-widget=\"insights\"]');
					if (embedContainer) {
						embedContainer.style.marginTop = '0';
						embedContainer.style.paddingTop = '0';
						var wrapper = embedContainer.closest('[data-embed-nav=\"true\"]');
						if (wrapper) {
							wrapper.style.marginTop = '0';
							wrapper.style.paddingTop = '0';
						}
					}
				}
			}

			// Run immediately and also on DOMContentLoaded
			hideTitleAndSpacing();
			if (document.readyState === 'loading') {
				document.addEventListener('DOMContentLoaded', hideTitleAndSpacing);
			}
		})();
		";
		wp_add_inline_script( 'freshnews-ai-embed-loader', $hide_title_script, 'before' );

		// Enqueue external embed script
		$script_args = array(
			'in_footer' => true,
		);

		// WordPress 6.3+ supports strategy parameter
		if ( isset( $options['defer_script_loading'] ) && $options['defer_script_loading'] ) {
			// Use strategy for WordPress 6.3+, fallback to filter for older versions
			if ( version_compare( get_bloginfo( 'version' ), '6.3', '>=' ) ) {
				$script_args['strategy'] = 'defer';
			} else {
				// For older WordPress versions, use filter to add defer attribute
				add_filter( 'script_loader_tag', array( $this, 'add_defer_attribute' ), 10, 2 );
			}
		}

		wp_enqueue_script(
			'freshnews-ai-embed-script',
			$embed_script_url,
			array(),
			FRESHNEWS_AI_CONTENT_HUB_VERSION,
			$script_args
		);

		// Enqueue local loader script
		wp_enqueue_script(
			'freshnews-ai-embed-loader',
			FRESHNEWS_AI_CONTENT_HUB_PLUGIN_URL . 'assets/js/embed-loader.js',
			array(),
			FRESHNEWS_AI_CONTENT_HUB_VERSION,
			true
		);

		// Pass configuration to loader script
		$signals_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
		wp_localize_script(
			'freshnews-ai-embed-loader',
			'freshnewsAiEmbedConfig',
			array(
				'newsroomUrl'      => $signals_url,
				'embedScriptUrl'   => $embed_script_url,
				'locale'           => isset( $options['language'] ) ? $options['language'] : 'en',
				'theme'            => isset( $options['theme'] ) ? $options['theme'] : 'auto',
				'canonicalUrlMode' => isset( $options['canonical_mode'] ) ? $options['canonical_mode'] : 'use_newsroom_url',
			)
		);
	}

	/**
	 * Add canonical link to head
	 */
	public function add_canonical_link() {
		// Only on pages where shortcode is present
		if ( ! $this->shortcode_present ) {
			return;
		}

		$options = get_option( 'freshnews_ai_newsroom_options', array() );

		// Check canonical mode
		if ( ! isset( $options['canonical_mode'] ) || 'use_newsroom_url' !== $options['canonical_mode'] ) {
			return;
		}

		// Check if newsroom URL is set
		$signals_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
		if ( empty( $signals_url ) ) {
			return;
		}

		// Output canonical link
		$canonical_url = esc_url( $signals_url );
		echo '<link rel="canonical" href="' . esc_url( $canonical_url ) . '" />' . "\n";
	}

	/**
	 * Maybe auto-create page (called when viewing settings page)
	 */
	public function maybe_auto_create_page() {
		// Only run on settings page
		if ( ! isset( $_GET['page'] ) || 'freshnews-ai-content-hub' !== $_GET['page'] ) {
			return;
		}

		// Check capability
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$this->create_pages_if_needed();
	}

	/**
	 * Auto-create pages when settings are saved
	 * Hooked to update_option_freshnews_ai_newsroom_options
	 *
	 * @param array $old_value Old option value
	 * @param array $value      New option value
	 */
	public function maybe_auto_create_page_on_save( $old_value, $value ) {
		// Check capability
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$this->create_pages_if_needed();
	}

	/**
	 * Create pages if auto-create is enabled
	 */
	private function create_pages_if_needed() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );

		// Check if auto-create is enabled
		if ( ! isset( $options['auto_create_page'] ) || ! $options['auto_create_page'] ) {
			return;
		}

		// Debug logging (only in development)
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'FreshNews.ai: create_pages_if_needed called. enable_insights_page: ' . ( ! empty( $options['enable_insights_page'] ) ? 'yes' : 'no' ) );
		}

		// Create/update Signals page
		if ( ! empty( $options['page_slug'] ) && ! empty( $options['page_title'] ) ) {
			$news_page = get_page_by_path( $options['page_slug'] );
			$news_page_data = array(
				'post_title'   => $options['page_title'],
				'post_name'    => $options['page_slug'],
				'post_content' => '[freshnews_newsroom mode="signals-list"]',
				'post_status'  => 'publish',
				'post_type'    => 'page',
			);

			if ( $news_page ) {
				// Update existing page
				$news_page_data['ID'] = $news_page->ID;
				$result = wp_update_post( $news_page_data );
				if ( is_wp_error( $result ) ) {
					error_log( 'FreshNews.ai: Failed to update Signals page: ' . $result->get_error_message() );
				}
			} else {
				// Create new page
				$result = wp_insert_post( $news_page_data );
				if ( is_wp_error( $result ) ) {
					error_log( 'FreshNews.ai: Failed to create Signals page: ' . $result->get_error_message() );
				} elseif ( $result ) {
					// Store success message in transient to show on next page load
					set_transient( 'freshnews_ai_news_page_created', true, 30 );
				}
			}
		}

		// Create/update Insights page (only if enabled)
		if ( ! empty( $options['enable_insights_page'] ) && ! empty( $options['insights_page_slug'] ) && ! empty( $options['insights_page_title'] ) ) {
			$insights_page = get_page_by_path( $options['insights_page_slug'] );
			$insights_page_data = array(
				'post_title'   => $options['insights_page_title'],
				'post_name'    => $options['insights_page_slug'],
				'post_content' => '[freshnews_newsroom mode="insights-list"]',
				'post_status'  => 'publish',
				'post_type'    => 'page',
			);

			if ( $insights_page ) {
				// Update existing page
				$insights_page_data['ID'] = $insights_page->ID;
				$result = wp_update_post( $insights_page_data );
				if ( is_wp_error( $result ) ) {
					// Log error but don't break
					error_log( 'FreshNews.ai: Failed to update Insights page: ' . $result->get_error_message() );
				}
			} else {
				// Create new page
				$result = wp_insert_post( $insights_page_data );
				if ( is_wp_error( $result ) ) {
					// Log error but don't break
					error_log( 'FreshNews.ai: Failed to create Insights page: ' . $result->get_error_message() );
				} elseif ( $result ) {
					// Store success message in transient to show on next page load
					set_transient( 'freshnews_ai_insights_page_created', true, 30 );
				}
			}
		} elseif ( ! empty( $options['enable_insights_page'] ) ) {
			// Insights is enabled but slug/title are missing - this shouldn't happen, but handle gracefully
		} else {
			// Insights page is disabled - delete any existing insights page if it exists
			$insights_slug = ! empty( $options['insights_page_slug'] ) ? $options['insights_page_slug'] : 'insights';
			$insights_page = get_page_by_path( $insights_slug );
			if ( $insights_page && strpos( $insights_page->post_content, '[freshnews_newsroom' ) !== false ) {
				// Only delete if it's a FreshNews page (safety check)
				wp_delete_post( $insights_page->ID, true );
			}
		}

		// Create/update FAQ page (only if enabled). When disabled, do not delete the page.
		if ( ! empty( $options['enable_faq_page'] ) ) {
			$faq_slug = 'faq';
			$faq_page = get_page_by_path( $faq_slug );
			$faq_page_data = array(
				'post_title'   => 'FAQ',
				'post_name'    => $faq_slug,
				'post_content' => '[freshnews_faq]',
				'post_status'  => 'publish',
				'post_type'    => 'page',
			);

			if ( $faq_page ) {
				// Update if page was created by our plugin (has shortcode) or content is empty so we can inject it
				$has_our_shortcode = strpos( $faq_page->post_content, '[freshnews_faq]' ) !== false;
				$content_empty     = trim( (string) $faq_page->post_content ) === '';
				if ( $has_our_shortcode || $content_empty ) {
					$faq_page_data['ID'] = $faq_page->ID;
					$result = wp_update_post( $faq_page_data );
					if ( is_wp_error( $result ) ) {
						error_log( 'FreshNews.ai: Failed to update FAQ page: ' . $result->get_error_message() );
					} elseif ( $content_empty && $result ) {
						set_transient( 'freshnews_ai_faq_page_created', true, 30 );
					}
				}
			} else {
				$result = wp_insert_post( $faq_page_data );
				if ( is_wp_error( $result ) ) {
					error_log( 'FreshNews.ai: Failed to create FAQ page: ' . $result->get_error_message() );
				} elseif ( $result ) {
					set_transient( 'freshnews_ai_faq_page_created', true, 30 );
				}
			}
		}
	}

	/**
	 * Add defer attribute to script tag (for WordPress < 6.3)
	 *
	 * @param string $tag Script tag HTML.
	 * @param string $handle Script handle.
	 * @return string Modified script tag.
	 */
	public function add_defer_attribute( $tag, $handle ) {
		if ( 'freshnews-ai-embed-script' === $handle ) {
			// Add defer attribute if not already present
			if ( false === strpos( $tag, ' defer' ) ) {
				$tag = str_replace( ' src', ' defer src', $tag );
			}
		}
		return $tag;
	}

	/**
	 * Add body class when shortcode is present
	 *
	 * @param array $classes Body classes.
	 * @return array Modified classes.
	 */
	public function add_body_class( $classes ) {
		if ( $this->shortcode_present || $this->faq_shortcode_present ) {
			$classes[] = 'has-freshnews-embed';
		}
		return $classes;
	}

	/**
	 * Hide page title when shortcode is present
	 *
	 * @param string $title The page title.
	 * @param int    $post_id The post ID.
	 * @return string Modified title (empty if shortcode is present).
	 */
	public function hide_page_title( $title, $post_id = null ) {
		// Only hide title if shortcode is present and we're on a page
		if ( ! $this->shortcode_present || ! is_page() ) {
			return $title;
		}

		// Get current page ID
		$current_page_id = get_the_ID();

		// Check if this is the current page being viewed
		if ( ( $post_id && $current_page_id === $post_id ) || ( ! $post_id && $current_page_id ) ) {
			// Check if this page contains the shortcode
			$post = get_post( $current_page_id );
			if ( $post && has_shortcode( $post->post_content, 'freshnews_newsroom' ) ) {
				// Return empty string to hide title, but only in the loop (not in nav menus, etc.)
				if ( in_the_loop() || is_main_query() ) {
					return ''; // Return empty string to hide title
				}
			}
		}

		return $title;
	}

	/**
	 * Add CSS to hide page title and reduce whitespace when shortcode is present
	 */
	public function add_hide_title_css() {
		// Only add CSS if shortcode is present
		if ( ! $this->shortcode_present ) {
			return;
		}

		?>
		<style type="text/css">
			/* Hide WordPress page title when FreshNews.ai shortcode is present */
			/* Use body class for more specific targeting */
			body.has-freshnews-embed .page .entry-title,
			body.has-freshnews-embed .page h1.entry-title,
			body.has-freshnews-embed .page .page-title,
			body.has-freshnews-embed .page h1.page-title,
			body.has-freshnews-embed .single .entry-title,
			body.has-freshnews-embed .single h1.entry-title,
			body.has-freshnews-embed .entry-header .entry-title,
			body.has-freshnews-embed .entry-header h1,
			body.has-freshnews-embed .post-title,
			body.has-freshnews-embed h1.post-title,
			body.has-freshnews-embed .page-header h1,
			body.has-freshnews-embed .page-header .page-title,
			body.has-freshnews-embed .entry-header,
			body.has-freshnews-embed .page-title,
			body.has-freshnews-embed h1.page-title,
			body.has-freshnews-embed .page h1:first-of-type,
			body.has-freshnews-embed .single h1:first-of-type,
			body.has-freshnews-embed .entry-title:first-of-type,
			body.has-freshnews-embed h1.entry-title:first-of-type,
			/* Common theme selectors */
			body.has-freshnews-embed .wp-block-post-title,
			body.has-freshnews-embed .has-global-padding h1:first-of-type,
			body.has-freshnews-embed .site-content h1:first-of-type,
			body.has-freshnews-embed .content-area h1:first-of-type,
			body.has-freshnews-embed #content h1:first-of-type,
			body.has-freshnews-embed main h1:first-of-type,
			body.has-freshnews-embed article h1:first-of-type,
			body.has-freshnews-embed .site-main h1:first-of-type,
			body.has-freshnews-embed .wp-block-post-content h1:first-of-type {
				display: none !important;
				margin: 0 !important;
				padding: 0 !important;
				height: 0 !important;
				overflow: hidden !important;
				line-height: 0 !important;
			}

			/* Reduce top whitespace - remove margins from entry-header and content area */
			body.has-freshnews-embed .page .entry-header,
			body.has-freshnews-embed .single .entry-header,
			body.has-freshnews-embed .page .entry-content:first-child,
			body.has-freshnews-embed .page #content .entry-content:first-child,
			body.has-freshnews-embed .page .content-area .entry-content:first-child,
			body.has-freshnews-embed .page .site-main,
			body.has-freshnews-embed .page main,
			body.has-freshnews-embed .page article,
			body.has-freshnews-embed .page .wp-block-post-content,
			body.has-freshnews-embed .page .has-global-padding,
			body.has-freshnews-embed .page .site-content,
			body.has-freshnews-embed .page #content {
				margin-top: 0 !important;
				padding-top: 0 !important;
			}

			/* Reduce whitespace above FreshNews embed container */
			body.has-freshnews-embed #freshnews-ai-root,
			body.has-freshnews-embed [data-freshnews-widget="insights"],
			body.has-freshnews-embed [data-embed-nav="true"] {
				margin-top: 0 !important;
				padding-top: 0 !important;
			}

			/* Remove top margin from wrapper if present */
			body.has-freshnews-embed [data-embed-nav="true"] {
				margin-top: 0 !important;
				padding-top: 0 !important;
			}

			/* Ensure embed container starts at top */
			body.has-freshnews-embed .fn-embed,
			body.has-freshnews-embed .fn-embed-signals-list,
			body.has-freshnews-embed .fn-embed-insights-list,
			body.has-freshnews-embed .fn-embed-faq-list {
				margin-top: 0 !important;
				padding-top: 0 !important;
			}

			/* Remove spacing from parent containers */
			body.has-freshnews-embed .page .entry-content,
			body.has-freshnews-embed .page .post-content,
			body.has-freshnews-embed .page .content {
				margin-top: 0 !important;
				padding-top: 0 !important;
			}

			/* Remove any spacing before the first element in content */
			body.has-freshnews-embed .entry-content > *:first-child,
			body.has-freshnews-embed .post-content > *:first-child,
			body.has-freshnews-embed .content > *:first-child {
				margin-top: 0 !important;
				padding-top: 0 !important;
			}
		</style>
		<?php
	}

	/**
	 * Display admin notices
	 */
	public function admin_notices() {
		// Only show to admins
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$options = get_option( 'freshnews_ai_newsroom_options', array() );

		// Show notice if plugin is enabled but not configured
		if ( isset( $options['enabled'] ) && $options['enabled'] ) {
			$signals_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
			if ( empty( $signals_url ) ) {
				$settings_url = admin_url( 'options-general.php?page=freshnews-ai-content-hub' );
				?>
				<div class="notice notice-warning is-dismissible">
					<p>
						<strong><?php esc_html_e( 'FreshNews AI Content Hub', 'freshnews-ai-content-hub' ); ?>:</strong>
						<?php esc_html_e( 'Please configure the plugin settings.', 'freshnews-ai-content-hub' ); ?>
						<a href="<?php echo esc_url( $settings_url ); ?>"><?php esc_html_e( 'Go to Settings', 'freshnews-ai-content-hub' ); ?></a>
					</p>
				</div>
				<?php
			}
		}

		// Show success notice if pages were just created
		if ( get_transient( 'freshnews_ai_news_page_created' ) ) {
			delete_transient( 'freshnews_ai_news_page_created' );
			?>
			<div class="notice notice-success is-dismissible">
				<p><strong><?php esc_html_e( 'FreshNews.ai:', 'freshnews-ai-content-hub' ); ?></strong> <?php esc_html_e( 'Signals page created successfully!', 'freshnews-ai-content-hub' ); ?></p>
			</div>
			<?php
		}

		if ( get_transient( 'freshnews_ai_insights_page_created' ) ) {
			delete_transient( 'freshnews_ai_insights_page_created' );
			?>
			<div class="notice notice-success is-dismissible">
				<p><strong><?php esc_html_e( 'FreshNews.ai:', 'freshnews-ai-content-hub' ); ?></strong> <?php esc_html_e( 'Insights page created successfully!', 'freshnews-ai-content-hub' ); ?></p>
			</div>
			<?php
		}

		if ( get_transient( 'freshnews_ai_faq_page_created' ) ) {
			delete_transient( 'freshnews_ai_faq_page_created' );
			?>
			<div class="notice notice-success is-dismissible">
				<p><strong><?php esc_html_e( 'FreshNews.ai:', 'freshnews-ai-content-hub' ); ?></strong> <?php esc_html_e( 'FAQ page created successfully!', 'freshnews-ai-content-hub' ); ?></p>
			</div>
			<?php
		}
	}

	/**
	 * Add settings link to plugins page
	 *
	 * @param array $links Existing plugin action links.
	 * @return array Modified links.
	 */
	public function add_plugin_action_links( $links ) {
		$settings_link = '<a href="' . admin_url( 'options-general.php?page=freshnews-ai-content-hub' ) . '">' . __( 'Settings', 'freshnews-ai-content-hub' ) . '</a>';
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Add rewrite rules for article URLs
	 * Handles URLs like /signals/[slug] and /insights/[slug]
	 */
	public function add_rewrite_rules() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );

		// Only add rewrite rules if plugin is enabled and configured
		$signals_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
		if ( empty( $options['enabled'] ) || empty( $signals_url ) ) {
			return;
		}

		// Get page slugs from settings
		$signals_slug = ! empty( $options['page_slug'] ) ? $options['page_slug'] : 'signals';
		$signals_slug = sanitize_title( $signals_slug );

		$insights_slug = ! empty( $options['insights_page_slug'] ) ? $options['insights_page_slug'] : 'insights';
		$insights_slug = sanitize_title( $insights_slug );

		// Add rewrite rule for Signals article URLs: /signals/[slug] and /en/signals/[slug]
		// This allows URLs like /signals/daily-news-2596903e or /en/signals/daily-news-2596903e to be handled
		add_rewrite_rule(
			'^' . $signals_slug . '/([^/]+)/?$',
			'index.php?freshnews_article_slug=$matches[1]&freshnews_page_type=signals',
			'top'
		);
		// Also handle /en/signals/[slug] format (language prefix)
		add_rewrite_rule(
			'^en/' . $signals_slug . '/([^/]+)/?$',
			'index.php?freshnews_article_slug=$matches[1]&freshnews_page_type=signals',
			'top'
		);

		// Add rewrite rule for Insights article URLs: /insights/[slug] and /en/insights/[slug]
		// This allows URLs like /insights/article-slug or /en/insights/article-slug to be handled
		// Only add if Insights page is enabled
		if ( ! empty( $options['enable_insights_page'] ) && $insights_slug !== $signals_slug ) {
			add_rewrite_rule(
				'^' . $insights_slug . '/([^/]+)/?$',
				'index.php?freshnews_article_slug=$matches[1]&freshnews_page_type=insights',
				'top'
			);
			// Also handle /en/insights/[slug] format (language prefix)
			add_rewrite_rule(
				'^en/' . $insights_slug . '/([^/]+)/?$',
				'index.php?freshnews_article_slug=$matches[1]&freshnews_page_type=insights',
				'top'
			);
		}

		// Virtual endpoint for AI discovery: /llms.txt (only when enabled and Signals URL set)
		if ( ! empty( $options['enable_llms_txt'] ) && ! empty( $signals_url ) ) {
			add_rewrite_rule( '^llms\.txt$', 'index.php?freshnews_llms=1', 'top' );
			add_rewrite_tag( '%freshnews_llms%', '1' );
		}

		// Flush rewrite rules on activation (handled in main plugin file)
		// But we need to check if rules need flushing
		$rules = get_option( 'rewrite_rules' );
		if ( ! isset( $rules['^' . $signals_slug . '/([^/]+)/?$'] ) ) {
			// Rules need flushing, but we can't do it on every request
			// Instead, we'll handle it via template_redirect
		}
	}

	/**
	 * Register custom query vars for virtual endpoints.
	 *
	 * @param array $vars Existing query vars.
	 * @return array Modified query vars.
	 */
	public function add_query_vars( $vars ) {
		$vars[] = 'freshnews_llms';
		$vars[] = 'freshnews_article_slug';
		$vars[] = 'freshnews_page_type';
		return $vars;
	}

	/**
	 * Resolve language code for llms.txt from plugin Language/Locale setting.
	 * If setting begins with "he" => "he"; else first part before "-" (e.g. en-US -> en); default "en".
	 *
	 * @return string Two-letter (or "he") language code.
	 */
	public function get_llms_txt_lang() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );
		$raw     = isset( $options['language'] ) ? trim( (string) $options['language'] ) : '';
		if ( $raw === '' ) {
			return 'en';
		}
		$raw = preg_replace( '/[^a-zA-Z-]/', '', $raw );
		if ( strpos( strtolower( $raw ), 'he' ) === 0 ) {
			return 'he';
		}
		$part = explode( '-', $raw )[0];
		return ( $part !== '' ) ? strtolower( $part ) : 'en';
	}

	/**
	 * Serve /llms.txt for AI discovery (plain text). Only when Enable llms.txt is on and Signals URL is set.
	 * Fetches mainsite llms.txt from Signals Hub to reuse the same generator logic.
	 */
	public function handle_llms_txt() {
		if ( ! get_query_var( 'freshnews_llms' ) ) {
			return;
		}
		$options      = get_option( 'freshnews_ai_newsroom_options', array() );
		if ( empty( $options['enable_llms_txt'] ) ) {
			return;
		}
		$signals_base = $this->get_signals_hub_base_url();
		if ( empty( $signals_base ) ) {
			return;
		}
		$base = rtrim( $signals_base, '/' );
		
		// Fetch mainsite llms.txt from Signals Hub (reuses same generator logic)
		$mainsite_llms_url = $base . '/mainsite/llms.txt';
		$response = wp_remote_get(
			$mainsite_llms_url,
			array(
				'timeout'    => 10,
				'sslverify'  => true,
				'user-agent' => 'FreshNews-AI-Content-Hub-Plugin/' . FRESHNEWS_AI_CONTENT_HUB_VERSION,
			)
		);
		
		if ( is_wp_error( $response ) ) {
			// Fallback to simple format if fetch fails
			$lang = $this->get_llms_txt_lang();
			$body = "# Signals Hub\n";
			$body .= "Signals-Authority-Hub: " . $base . "/llms.txt\n";
			$body .= "Latest-Signals: " . $base . "/" . $lang . "/latest\n";
			$body .= "Signals-Sitemap: " . $base . "/sitemap.xml\n";
			$body .= "Signals-RSS: " . $base . "/rss.xml\n";
		} else {
			$code = wp_remote_retrieve_response_code( $response );
			if ( $code === 200 ) {
				$body = wp_remote_retrieve_body( $response );
				if ( ! is_string( $body ) || empty( $body ) ) {
					// Fallback if body is empty
					$lang = $this->get_llms_txt_lang();
					$body = "# Signals Hub\n";
					$body .= "Signals-Authority-Hub: " . $base . "/llms.txt\n";
					$body .= "Latest-Signals: " . $base . "/" . $lang . "/latest\n";
					$body .= "Signals-Sitemap: " . $base . "/sitemap.xml\n";
					$body .= "Signals-RSS: " . $base . "/rss.xml\n";
				}
			} else {
				// Fallback if non-200 response
				$lang = $this->get_llms_txt_lang();
				$body = "# Signals Hub\n";
				$body .= "Signals-Authority-Hub: " . $base . "/llms.txt\n";
				$body .= "Latest-Signals: " . $base . "/" . $lang . "/latest\n";
				$body .= "Signals-Sitemap: " . $base . "/sitemap.xml\n";
				$body .= "Signals-RSS: " . $base . "/rss.xml\n";
			}
		}
		
		header( 'Content-Type: text/plain; charset=utf-8' );
		header( 'Cache-Control: public, s-maxage=3600, stale-while-revalidate=86400' );
		echo $body; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- plain text, no HTML
		exit;
	}

	/**
	 * Handle article URLs early in parse_request to prevent 404
	 * This runs before WordPress decides what to show
	 */
	public function handle_article_url_early( $wp ) {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );

		// Only handle if plugin is enabled and configured
		$signals_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
		if ( empty( $options['enabled'] ) || empty( $signals_url ) ) {
			return;
		}

		// Get page slugs from settings
		$signals_slug = ! empty( $options['page_slug'] ) ? $options['page_slug'] : 'signals';
		$signals_slug = sanitize_title( $signals_slug );

		$insights_enabled = ! empty( $options['enable_insights_page'] );
		$insights_slug = $insights_enabled && ! empty( $options['insights_page_slug'] ) ? $options['insights_page_slug'] : 'insights';
		$insights_slug = sanitize_title( $insights_slug );

		// Get current request URI
		$request_uri = isset( $_SERVER['REQUEST_URI'] ) ? $_SERVER['REQUEST_URI'] : '';
		$request_uri = strtok( $request_uri, '?' ); // Remove query string
		$request_uri = trim( $request_uri, '/' );

		// Check if URL matches /signals/[slug] or /en/signals/[slug] pattern
		$signals_pattern = '/^' . preg_quote( $signals_slug, '/' ) . '\/([^\/]+)$/';
		$signals_pattern_with_lang = '/^en\/' . preg_quote( $signals_slug, '/' ) . '\/([^\/]+)$/';
		$insights_pattern = '/^' . preg_quote( $insights_slug, '/' ) . '\/([^\/]+)$/';
		$insights_pattern_with_lang = '/^en\/' . preg_quote( $insights_slug, '/' ) . '\/([^\/]+)$/';

		$matches = array();
		if ( preg_match( $signals_pattern, $request_uri, $matches ) || preg_match( $signals_pattern_with_lang, $request_uri, $matches ) ) {
			// Set query vars so WordPress recognizes this as a valid request
			$wp->query_vars['freshnews_article_slug'] = sanitize_text_field( $matches[1] );
			$wp->query_vars['freshnews_page_type'] = 'signals';
		} elseif ( $insights_enabled && ( preg_match( $insights_pattern, $request_uri, $matches ) || preg_match( $insights_pattern_with_lang, $request_uri, $matches ) ) ) {
			// Set query vars so WordPress recognizes this as a valid request
			$wp->query_vars['freshnews_article_slug'] = sanitize_text_field( $matches[1] );
			$wp->query_vars['freshnews_page_type'] = 'insights';
		}
	}

	/**
	 * Handle template redirect for article URLs
	 * Detects /signals/[slug] URLs and ensures the shortcode page is loaded
	 */
	public function handle_article_url() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );

		// Only handle if plugin is enabled and configured
		$signals_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
		if ( empty( $options['enabled'] ) || empty( $signals_url ) ) {
			return;
		}

		// Get page slugs from settings
		$signals_slug = ! empty( $options['page_slug'] ) ? $options['page_slug'] : 'signals';
		$signals_slug = sanitize_title( $signals_slug );

		$insights_enabled = ! empty( $options['enable_insights_page'] );
		$insights_slug = $insights_enabled && ! empty( $options['insights_page_slug'] ) ? $options['insights_page_slug'] : 'insights';
		$insights_slug = sanitize_title( $insights_slug );

		// First, check if WordPress recognized the rewrite rule (query vars are set)
		global $wp_query;
		$article_slug = get_query_var( 'freshnews_article_slug' );
		$page_type_from_query = get_query_var( 'freshnews_page_type' );

		// If query vars are set, use them
		if ( ! empty( $article_slug ) && ! empty( $page_type_from_query ) ) {
			$page_type = $page_type_from_query;
			$page_slug = ( $page_type === 'insights' ) ? $insights_slug : $signals_slug;
		} else {
			// Fallback: manually parse the URL (for cases where rewrite rules haven't been flushed)
			// Get current request URI
			$request_uri = isset( $_SERVER['REQUEST_URI'] ) ? $_SERVER['REQUEST_URI'] : '';
			$request_uri = strtok( $request_uri, '?' ); // Remove query string
			$request_uri = trim( $request_uri, '/' );

			// Check if URL matches /signals/[slug] or /en/signals/[slug] pattern
			$signals_pattern = '/^' . preg_quote( $signals_slug, '/' ) . '\/([^\/]+)$/';
			$signals_pattern_with_lang = '/^en\/' . preg_quote( $signals_slug, '/' ) . '\/([^\/]+)$/';
			$insights_pattern = '/^' . preg_quote( $insights_slug, '/' ) . '\/([^\/]+)$/';
			$insights_pattern_with_lang = '/^en\/' . preg_quote( $insights_slug, '/' ) . '\/([^\/]+)$/';

			$page_slug = null;
			$page_type = null;
			$matches = array();

			if ( preg_match( $signals_pattern, $request_uri, $matches ) || preg_match( $signals_pattern_with_lang, $request_uri, $matches ) ) {
				$page_slug = $signals_slug;
				$page_type = 'signals';
			} elseif ( $insights_enabled && ( preg_match( $insights_pattern, $request_uri, $matches ) || preg_match( $insights_pattern_with_lang, $request_uri, $matches ) ) ) {
				$page_slug = $insights_slug;
				$page_type = 'insights';
			}

			if ( $page_slug && isset( $matches[1] ) ) {
				$article_slug = sanitize_text_field( $matches[1] );
			} else {
				// No match found
				return;
			}
		}

		if ( ! empty( $article_slug ) && ! empty( $page_slug ) ) {

			// Special handling for "about" page - it should be rendered like an article
			// but with different content (about page from FreshNews domain)
			if ( $article_slug === 'about' ) {
				// Store "about" as the article slug so embed script can detect it
				$this->current_article_slug = 'about';
			} else {
				// Store the article slug for use in other methods
				$this->current_article_slug = $article_slug;
			}

			// Get the page that contains the shortcode
			$page_id = null;
			if ( ! empty( $options['auto_create_page'] ) && ! empty( $page_slug ) ) {
				// Find page by slug
				$page = get_page_by_path( $page_slug );
				if ( $page ) {
					$page_id = $page->ID;
				}
			}

			// If no page found, try to find any page with the shortcode matching the page type
			if ( ! $page_id ) {
				global $wpdb;
				$mode = ( $page_type === 'insights' ) ? 'insights-list' : 'signals-list';
				$page_id = $wpdb->get_var(
					$wpdb->prepare(
						"SELECT ID FROM {$wpdb->posts}
						WHERE post_type = 'page'
						AND post_status = 'publish'
						AND post_content LIKE %s
						LIMIT 1",
						'%[freshnews_newsroom%' . $mode . '%'
					)
				);

				// Fallback: find any page with the shortcode
				if ( ! $page_id ) {
					$page_id = $wpdb->get_var(
						$wpdb->prepare(
							"SELECT ID FROM {$wpdb->posts}
							WHERE post_type = 'page'
							AND post_status = 'publish'
							AND post_content LIKE %s
							LIMIT 1",
							'%[freshnews_newsroom%'
						)
					);
				}
			}

			// If we found a page, make sure it loads (don't redirect, just ensure it's the right page)
			// The embed script will detect the slug from the URL path and render the article
			if ( $page_id ) {
				// Set up the query to load the page
				global $wp_query;
				
				// Prevent 404
				$wp_query->is_404 = false;
				$wp_query->is_page = true;
				$wp_query->is_singular = true;
				$wp_query->is_home = false;
				$wp_query->is_front_page = false;
				
				// Set the queried object
				$wp_query->queried_object = get_post( $page_id );
				$wp_query->queried_object_id = $page_id;
				
				// Set the page as the current post
				$GLOBALS['post'] = get_post( $page_id );
				setup_postdata( $GLOBALS['post'] );
				
				// Ensure WordPress uses the correct template
				$wp_query->posts = array( $GLOBALS['post'] );
				$wp_query->post_count = 1;
				$wp_query->found_posts = 1;
			}
		}
	}

	/**
	 * Register custom sitemap provider for FreshNews.ai articles
	 * Integrates with WordPress 5.5+ sitemap API
	 */
	public function register_sitemap_provider() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );

		// Only register if plugin is enabled
		$signals_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
		if ( empty( $options['enabled'] ) || empty( $signals_url ) ) {
			return;
		}

		// Only register if class exists (WordPress 5.5+)
		if ( class_exists( 'FreshNews_AI_Sitemap_Provider' ) ) {
			wp_sitemaps_register_provider( 'freshnews-articles', new FreshNews_AI_Sitemap_Provider() );
		}
	}

	/**
	 * Register WordPress widget
	 */
	public function register_widget() {
		if ( class_exists( 'FreshNews_AI_Widget' ) ) {
			register_widget( 'FreshNews_AI_Widget' );
		}
	}

	/**
	 * Add JSON-LD schema for articles (E-E-A-T optimization)
	 * Injects NewsArticle schema into head with proper publisher and mainEntityOfPage
	 */
	public function add_jsonld_schema() {
		$options = get_option( 'freshnews_ai_newsroom_options', array() );

		// Check if plugin is enabled and configured
		$signals_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
		if ( empty( $options['enabled'] ) || empty( $signals_url ) ) {
			return;
		}

		// Get article slug (from property or extract from URL)
		$article_slug = $this->current_article_slug;
		if ( ! $article_slug ) {
			// Fallback: extract from URL
			$request_uri = isset( $_SERVER['REQUEST_URI'] ) ? $_SERVER['REQUEST_URI'] : '';
			$request_uri = strtok( $request_uri, '?' ); // Remove query string
			$request_uri = trim( $request_uri, '/' );

			$page_slug = ! empty( $options['page_slug'] ) ? $options['page_slug'] : 'signals';
			$page_slug = sanitize_title( $page_slug );
			$pattern = '/^' . preg_quote( $page_slug, '/' ) . '\/([^\/]+)$/';
			if ( preg_match( $pattern, $request_uri, $matches ) ) {
				$article_slug = sanitize_text_field( $matches[1] );
			}
		}

		// Only on article pages
		if ( ! $article_slug ) {
			return;
		}

		// Fetch article data from API
		$article_data = $this->fetch_article_data( $article_slug, $options );
		if ( ! $article_data ) {
			return;
		}

		// Store slug for schema generation
		$this->current_article_slug = $article_slug;

		// Generate JSON-LD schema
		$schema = $this->generate_article_schema( $article_data, $options );

		// Output schema
		if ( ! empty( $schema ) ) {
			echo '<script type="application/ld+json">' . "\n";
			echo wp_json_encode( $schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT ) . "\n";
			echo '</script>' . "\n";
		}
	}

	/**
	 * Fetch article data from FreshNews.ai API
	 *
	 * @param string $slug Article slug
	 * @param array  $options Plugin options
	 * @return array|null Article data or null on failure
	 */
	private function fetch_article_data( $slug, $options ) {
		$newsroom_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
		if ( empty( $newsroom_url ) ) {
			return null;
		}
		$language = ! empty( $options['language'] ) ? $options['language'] : 'en';

		// Extract domain from newsroom URL
		$parsed_url = wp_parse_url( $newsroom_url );
		if ( ! $parsed_url || empty( $parsed_url['host'] ) ) {
			return null;
		}

		$domain = $parsed_url['host'];
		$api_base = $parsed_url['scheme'] . '://' . $parsed_url['host'];
		if ( ! empty( $parsed_url['port'] ) ) {
			$api_base .= ':' . $parsed_url['port'];
		}

		// Build API endpoint: /api/public/[domain]/[lang]/articles/[slug]
		$api_url = $api_base . '/api/public/' . rawurlencode( $domain ) . '/' . rawurlencode( $language ) . '/articles/' . rawurlencode( $slug );

		// Fetch with timeout
		$response = wp_remote_get(
			$api_url,
			array(
				'timeout'     => 10,
				'headers'     => array(
					'Accept' => 'application/json',
				),
			)
		);

		if ( is_wp_error( $response ) ) {
			return null;
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		if ( 200 !== $status_code ) {
			return null;
		}

		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		// API might return { article: {...} } or article directly
		return isset( $data['article'] ) ? $data['article'] : $data;
	}

	/**
	 * Generate JSON-LD schema for article
	 *
	 * @param array $article Article data from API
	 * @param array $options Plugin options
	 * @return array JSON-LD schema array
	 */
	private function generate_article_schema( $article, $options ) {
		$newsroom_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
		if ( empty( $newsroom_url ) ) {
			return array();
		}
		$site_url = home_url();

		// Get site name (WordPress site name)
		$site_name = get_bloginfo( 'name' );
		if ( empty( $site_name ) ) {
			$site_name = wp_parse_url( $site_url, PHP_URL_HOST );
		}

		// Build article URL on CNAME subdomain (mainEntityOfPage)
		$article_slug = $this->current_article_slug;
		if ( ! $article_slug ) {
			// Fallback: try to get from article data
			$article_slug = isset( $article['slug'] ) ? $article['slug'] : '';
		}
		$article_url = $newsroom_url . '/signals/' . rawurlencode( $article_slug );

		// Extract article data
		$title = isset( $article['title'] ) ? $article['title'] : '';
		$description = isset( $article['dek'] ) ? $article['dek'] : ( isset( $article['excerpt'] ) ? $article['excerpt'] : '' );
		$published_date = isset( $article['publishedAt'] ) ? $article['publishedAt'] : ( isset( $article['published_at'] ) ? $article['published_at'] : '' );
		$updated_date = isset( $article['updatedAt'] ) ? $article['updatedAt'] : ( isset( $article['updated_at'] ) ? $article['updated_at'] : '' );

		// Get hero image
		$image_url = null;
		if ( isset( $article['heroImage']['url'] ) ) {
			$image_url = $article['heroImage']['url'];
		} elseif ( isset( $article['hero']['image']['url'] ) ) {
			$image_url = $article['hero']['image']['url'];
		} elseif ( isset( $article['heroImageUrl'] ) ) {
			$image_url = $article['heroImageUrl'];
		}

		// Build schema
		$schema = array(
			'@context' => 'https://schema.org',
			'@type'    => 'NewsArticle',
			'headline' => $title,
			'url'      => $article_url,
			'mainEntityOfPage' => array(
				'@type' => 'WebPage',
				'@id'   => $article_url,
			),
			'publisher' => array(
				'@type' => 'Organization',
				'name'  => $site_name,
				'url'   => $site_url,
			),
		);

		// Add description
		if ( ! empty( $description ) ) {
			$schema['description'] = wp_strip_all_tags( $description );
		}

		// Add dates
		if ( ! empty( $published_date ) ) {
			$schema['datePublished'] = gmdate( 'c', strtotime( $published_date ) );
		}
		if ( ! empty( $updated_date ) ) {
			$schema['dateModified'] = gmdate( 'c', strtotime( $updated_date ) );
		} elseif ( ! empty( $published_date ) ) {
			// Fallback to published date if no updated date
			$schema['dateModified'] = gmdate( 'c', strtotime( $published_date ) );
		}

		// Add image
		if ( ! empty( $image_url ) ) {
			$schema['image'] = array(
				'@type' => 'ImageObject',
				'url'   => esc_url_raw( $image_url ),
			);
		}

		// Add author/persona if available
		if ( isset( $article['perspective'] ) && is_array( $article['perspective'] ) ) {
			$persona = $article['perspective'];
			$author_name = null;
			if ( isset( $persona['displayName'] ) ) {
				$author_name = $persona['displayName'];
			} elseif ( isset( $persona['personaName'] ) ) {
				$author_name = $persona['personaName'];
			} elseif ( isset( $persona['attribution'] ) ) {
				$author_name = $persona['attribution'];
			}

			if ( $author_name ) {
				$schema['author'] = array(
					'@type' => 'Person',
					'name'  => $author_name,
				);
			}
		}

		return $schema;
	}

	/**
	 * Add JavaScript to display author block for articles (E-E-A-T optimization)
	 * Links persona information to WordPress author pages
	 */
	public function add_author_block_script() {
		// Only on pages where shortcode is present
		if ( ! $this->shortcode_present ) {
			return;
		}

		$options = get_option( 'freshnews_ai_newsroom_options', array() );

		// Only if plugin is enabled
		$signals_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
		if ( empty( $options['enabled'] ) || empty( $signals_url ) ) {
			return;
		}

		?>
		<script type="text/javascript">
		(function() {
			'use strict';

			// Wait for DOM and embed to be ready
			function initAuthorBlock() {
				var container = document.getElementById('freshnews-ai-root');
				if (!container) {
					return;
				}

				// Check if we're on an article page (has article content)
				var articleContent = container.querySelector('.fn-embed-article, [class*="fn-embed-article"]');
				if (!articleContent) {
					// Not an article page, check periodically
					setTimeout(initAuthorBlock, 1000);
					return;
				}

				// Check if author block already exists
				if (container.querySelector('.freshnews-author-block')) {
					return;
				}

				// Try to get article data from embed script or fetch it
				var articleSlug = getArticleSlugFromUrl();
				if (!articleSlug) {
					return;
				}

				// Fetch article data to get persona
				fetchArticleForAuthor(articleSlug, container);
			}

			function getArticleSlugFromUrl() {
				var path = window.location.pathname;
				var parts = path.split('/').filter(function(p) { return p; });

				// Look for slug after 'signals' or 'insights'
				var signalsIndex = parts.indexOf('signals');
				var insightsIndex = parts.indexOf('insights');
				var index = signalsIndex !== -1 ? signalsIndex : insightsIndex;

				if (index !== -1 && index < parts.length - 1) {
					return parts[index + 1];
				}

				// Fallback: last part
				if (parts.length > 0) {
					var lastPart = parts[parts.length - 1];
					var skipParts = ['signals', 'insights', 'about', 'en', 'he', 'es', 'fr', 'de', 'it', 'pt', 'ja', 'zh', 'ar'];
					if (!skipParts.includes(lastPart) && lastPart.length > 0) {
						return lastPart;
					}
				}

				return null;
			}

			function fetchArticleForAuthor(slug, container) {
				var root = container.closest('[data-embed-nav]') || container;
				var apiBase = root.getAttribute('data-api-base') || container.getAttribute('data-api-base') || '';
				var language = root.getAttribute('data-language') || container.getAttribute('data-language') || 'en';

				if (!apiBase) {
					return;
				}

				// Extract domain from apiBase
				var domainMatch = apiBase.match(/https?:\/\/([^\/:]+)/);
				if (!domainMatch) {
					return;
				}

				var domain = domainMatch[1].split(':')[0];
				var apiUrl = apiBase.replace(/\/+$/, '') + '/api/public/' + encodeURIComponent(domain) + '/' + encodeURIComponent(language) + '/articles/' + encodeURIComponent(slug);

				fetch(apiUrl, {
					headers: {
						'Accept': 'application/json'
					}
				})
				.then(function(response) {
					if (!response.ok) {
						return null;
					}
					return response.json();
				})
				.then(function(data) {
					if (!data) {
						return;
					}

					var article = data.article || data;
					if (!article || !article.perspective) {
						return;
					}

					var persona = article.perspective;
					displayAuthorBlock(container, persona);
				})
				.catch(function(error) {
					// Silently fail
					console.debug('[FreshNews Author Block] Could not fetch article:', error);
				});
			}

			function displayAuthorBlock(container, persona) {
				var authorName = persona.displayName || persona.personaName || persona.attribution || null;
				if (!authorName) {
					return;
				}

				// Try to find WordPress user by name
				var authorUrl = findWordPressAuthorUrl(authorName);

				var block = document.createElement('div');
				block.className = 'freshnews-author-block';
				block.style.cssText = 'margin-top: 2rem; padding: 1.5rem; border-top: 1px solid #e5e7eb; background-color: #f9fafb;';

				var title = document.createElement('h3');
				title.textContent = 'About the Author';
				title.style.cssText = 'margin: 0 0 0.75rem 0; font-size: 1.125rem; font-weight: 600;';
				block.appendChild(title);

				var authorInfo = document.createElement('div');
				authorInfo.style.cssText = 'display: flex; align-items: center; gap: 1rem;';

				if (authorUrl) {
					var authorLink = document.createElement('a');
					authorLink.href = authorUrl;
					authorLink.textContent = authorName;
					authorLink.style.cssText = 'font-weight: 500; color: #2563eb; text-decoration: none;';
					authorInfo.appendChild(authorLink);
				} else {
					var authorSpan = document.createElement('span');
					authorSpan.textContent = authorName;
					authorSpan.style.cssText = 'font-weight: 500;';
					authorInfo.appendChild(authorSpan);
				}

				if (persona.personaTitle) {
					var titleSpan = document.createElement('span');
					titleSpan.textContent = persona.personaTitle;
					titleSpan.style.cssText = 'color: #6b7280; font-size: 0.875rem;';
					authorInfo.appendChild(titleSpan);
				}

				block.appendChild(authorInfo);

				// Insert after the embed container
				container.parentNode.insertBefore(block, container.nextSibling);
			}

			function findWordPressAuthorUrl(authorName) {
				// This would ideally be populated by PHP with WordPress author data
				// For now, we'll try to construct a URL based on common WordPress author URL patterns
				var authorSlug = authorName.toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/^-|-$/g, '');
				var baseUrl = '<?php echo esc_js( home_url() ); ?>';

				// Try common WordPress author URL patterns
				var possibleUrls = [
					baseUrl + '/author/' + authorSlug,
					baseUrl + '/authors/' + authorSlug,
				];

				// Return first URL (could be enhanced to check if URL exists)
				return possibleUrls[0];
			}

			// Initialize when DOM is ready
			if (document.readyState === 'loading') {
				document.addEventListener('DOMContentLoaded', function() {
					setTimeout(initAuthorBlock, 2000); // Wait for embed to render
				});
			} else {
				setTimeout(initAuthorBlock, 2000);
			}

			// Also check periodically in case article loads later
			var checkInterval = setInterval(function() {
				var container = document.getElementById('freshnews-ai-root');
				if (container && !container.querySelector('.freshnews-author-block')) {
					var articleContent = container.querySelector('.fn-embed-article, [class*="fn-embed-article"]');
					if (articleContent) {
						initAuthorBlock();
						clearInterval(checkInterval);
					}
				}
			}, 2000);

			// Stop checking after 30 seconds
			setTimeout(function() {
				clearInterval(checkInterval);
			}, 30000);
		})();
		</script>
		<?php
	}
}

/**
 * Custom sitemap provider for FreshNews.ai articles
 * Fetches articles from API and adds them to WordPress sitemap
 */
if ( ! function_exists( 'freshnews_ai_register_sitemap_provider_class' ) ) {
	function freshnews_ai_register_sitemap_provider_class() {
		if ( class_exists( 'WP_Sitemaps_Provider' ) && ! class_exists( 'FreshNews_AI_Sitemap_Provider' ) ) {
			class FreshNews_AI_Sitemap_Provider extends WP_Sitemaps_Provider {

				/**
				 * Provider name
				 *
				 * @var string
				 */
				public $name = 'freshnews-articles';

				/**
				 * Get max number of pages for the sitemap
				 *
				 * @param string $object_subtype Object subtype (unused for articles)
				 * @return int
				 */
				public function get_max_num_pages( $object_subtype = '' ) {
					$options = get_option( 'freshnews_ai_newsroom_options', array() );

					// Only include if plugin is enabled
					$signals_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
					if ( empty( $options['enabled'] ) || empty( $signals_url ) ) {
						return 0;
					}

					// Fetch articles to determine page count
					$articles = $this->fetch_articles( $options );
					if ( empty( $articles ) ) {
						return 0;
					}

					// WordPress sitemap limits to 50,000 URLs per sitemap, 50,000 per page
					// We'll use 1 page (all articles on one sitemap page)
					return 1;
				}

				/**
				 * Get URL list for sitemap (required abstract method)
				 *
				 * @param int    $page_num Page number
				 * @param string $object_subtype Object subtype (unused for articles)
				 * @return array Array of sitemap URL data
				 */
				public function get_url_list( $page_num, $object_subtype = '' ) {
					$options = get_option( 'freshnews_ai_newsroom_options', array() );

					// Only include if plugin is enabled
					$signals_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
					if ( empty( $options['enabled'] ) || empty( $signals_url ) ) {
						return array();
					}

					$page_slug = ! empty( $options['page_slug'] ) ? $options['page_slug'] : 'signals';
					$page_slug = sanitize_title( $page_slug );

					// Get the page that contains the shortcode
					$page = get_page_by_path( $page_slug );
					if ( ! $page ) {
						// Try to find any page with the shortcode
						global $wpdb;
						$page_id = $wpdb->get_var(
							$wpdb->prepare(
								"SELECT ID FROM {$wpdb->posts}
								WHERE post_type = 'page'
								AND post_status = 'publish'
								AND post_content LIKE %s
								LIMIT 1",
								'%[freshnews_newsroom%'
							)
						);
						if ( $page_id ) {
							$page = get_post( $page_id );
						}
					}

					if ( ! $page ) {
						return array();
					}

					$page_url = get_permalink( $page->ID );
					$base_url = rtrim( $page_url, '/' );

					// Fetch articles from API
					$articles = $this->fetch_articles( $options );
					if ( empty( $articles ) ) {
						return array();
					}

					$urls = array();

					foreach ( $articles as $article ) {
						$slug = isset( $article['slug'] ) ? $article['slug'] : '';
						if ( empty( $slug ) ) {
							continue;
						}

						// Build article URL on WordPress site
						$article_url = $base_url . '/' . rawurlencode( $slug );

						// Get last modified date
						$lastmod = null;
						if ( isset( $article['updatedAt'] ) ) {
							$lastmod = $article['updatedAt'];
						} elseif ( isset( $article['publishedAt'] ) ) {
							$lastmod = $article['publishedAt'];
						}

						$urls[] = array(
							'loc' => $article_url,
						);

						if ( $lastmod ) {
							$urls[ count( $urls ) - 1 ]['lastmod'] = gmdate( 'c', strtotime( $lastmod ) );
						}
					}

					return $urls;
				}

				/**
				 * Fetch articles from FreshNews.ai API
				 *
				 * @param array $options Plugin options
				 * @return array Array of articles
				 */
				private function fetch_articles( $options ) {
					$newsroom_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
					if ( empty( $newsroom_url ) ) {
						return array();
					}
					$language = ! empty( $options['language'] ) ? $options['language'] : 'en';

					// Extract domain from newsroom URL
					$parsed_url = wp_parse_url( $newsroom_url );
					if ( ! $parsed_url || empty( $parsed_url['host'] ) ) {
						return array();
					}

					$domain = $parsed_url['host'];
					$api_base = $parsed_url['scheme'] . '://' . $parsed_url['host'];
					if ( ! empty( $parsed_url['port'] ) ) {
						$api_base .= ':' . $parsed_url['port'];
					}

					// Build API endpoint: /api/public/[domain]/[lang]/signals
					$api_url = $api_base . '/api/public/' . rawurlencode( $domain ) . '/' . rawurlencode( $language ) . '/signals?limit=1000';

					// Fetch with timeout and caching
					$cache_key = 'freshnews_articles_sitemap_' . md5( $api_url );
					$cached = get_transient( $cache_key );
					if ( false !== $cached ) {
						return $cached;
					}

					$response = wp_remote_get(
						$api_url,
						array(
							'timeout' => 15,
							'headers' => array(
								'Accept' => 'application/json',
							),
						)
					);

					if ( is_wp_error( $response ) ) {
						return array();
					}

					$status_code = wp_remote_retrieve_response_code( $response );
					if ( 200 !== $status_code ) {
						return array();
					}

					$body = wp_remote_retrieve_body( $response );
					$data = json_decode( $body, true );

					// API might return array directly or { articles: [...] }
					$articles = is_array( $data ) && isset( $data[0] ) ? $data : ( isset( $data['articles'] ) ? $data['articles'] : array() );

					// Cache for 1 hour
					set_transient( $cache_key, $articles, HOUR_IN_SECONDS );

					return $articles;
				}
			} // End class FreshNews_AI_Sitemap_Provider
		} // End if class_exists WP_Sitemaps_Provider
	} // End function
	add_action( 'plugins_loaded', 'freshnews_ai_register_sitemap_provider_class', 5 );
}

/**
 * WordPress Widget for FreshNews AI Content Hub
 * Allows embedding content hub in sidebars and footers
 */
if ( ! function_exists( 'freshnews_ai_register_widget_class' ) ) {
	function freshnews_ai_register_widget_class() {
		if ( class_exists( 'WP_Widget' ) && ! class_exists( 'FreshNews_AI_Widget' ) ) {
			class FreshNews_AI_Widget extends WP_Widget {

				/**
				 * Constructor
				 */
				public function __construct() {
					parent::__construct(
						'freshnews_ai_widget',
						__( 'FreshNews AI Content Hub', 'freshnews-ai-content-hub' ),
						array(
							'description' => __( 'Display FreshNews.ai content hub in your sidebar or footer', 'freshnews-ai-content-hub' ),
						)
					);
				}

				/**
				 * Output widget content
				 *
				 * @param array $args     Widget arguments
				 * @param array $instance Widget instance
				 */
							public function widget( $args, $instance ) {
					$options = get_option( 'freshnews_ai_newsroom_options', array() );

					// Check if plugin is enabled
					$signals_url = function_exists( 'freshnews_get_signals_url' ) ? freshnews_get_signals_url() : '';
					if ( empty( $options['enabled'] ) || empty( $signals_url ) ) {
						if ( current_user_can( 'manage_options' ) ) {
							echo $args['before_widget'];
							echo $args['before_title'] . esc_html__( 'FreshNews AI Content Hub', 'freshnews-ai-content-hub' ) . $args['after_title'];
							echo '<p>' . esc_html__( 'Please configure FreshNews.ai in Settings → FreshNews AI Content Hub', 'freshnews-ai-content-hub' ) . '</p>';
							echo $args['after_widget'];
						}
						return;
					}

					// Get widget settings
					$title = ! empty( $instance['title'] ) ? $instance['title'] : '';
					$mode = ! empty( $instance['mode'] ) ? $instance['mode'] : 'signals-list';
					$kind = ! empty( $instance['kind'] ) ? $instance['kind'] : 'daily';
					$limit = ! empty( $instance['limit'] ) ? absint( $instance['limit'] ) : 5;
					$language = ! empty( $instance['language'] ) ? $instance['language'] : ( ! empty( $options['language'] ) ? $options['language'] : 'en' );

					// Build shortcode
					$shortcode = sprintf(
						'[freshnews_newsroom mode="%s" kind="%s" limit="%d" language="%s"]',
						esc_attr( $mode ),
						esc_attr( $kind ),
						$limit,
						esc_attr( $language )
					);

					// Output widget
					echo $args['before_widget'];

					if ( ! empty( $title ) ) {
						echo $args['before_title'] . esc_html( $title ) . $args['after_title'];
					}

					echo do_shortcode( $shortcode );

					echo $args['after_widget'];
				}

				/**
				 * Output widget form
				 *
				 * @param array $instance Widget instance
				 */
				public function form( $instance ) {
					$title = ! empty( $instance['title'] ) ? $instance['title'] : '';
					$mode = ! empty( $instance['mode'] ) ? $instance['mode'] : 'signals-list';
					$kind = ! empty( $instance['kind'] ) ? $instance['kind'] : 'daily';
					$limit = ! empty( $instance['limit'] ) ? absint( $instance['limit'] ) : 5;
					$language = ! empty( $instance['language'] ) ? $instance['language'] : 'en';

					$options = get_option( 'freshnews_ai_newsroom_options', array() );
					$default_language = ! empty( $options['language'] ) ? $options['language'] : 'en';
					?>
					<p>
						<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Title:', 'freshnews-ai-content-hub' ); ?></label>
						<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>">
					</p>
					<p>
						<label for="<?php echo esc_attr( $this->get_field_id( 'mode' ) ); ?>"><?php esc_html_e( 'Mode:', 'freshnews-ai-content-hub' ); ?></label>
						<select class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'mode' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'mode' ) ); ?>">
							<option value="signals-list" <?php selected( $mode, 'signals-list' ); ?>><?php esc_html_e( 'Signals List', 'freshnews-ai-content-hub' ); ?></option>
							<option value="insights-list" <?php selected( $mode, 'insights-list' ); ?>><?php esc_html_e( 'Insights List', 'freshnews-ai-content-hub' ); ?></option>
						</select>
					</p>
					<p>
						<label for="<?php echo esc_attr( $this->get_field_id( 'kind' ) ); ?>"><?php esc_html_e( 'Kind:', 'freshnews-ai-content-hub' ); ?></label>
						<select class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'kind' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'kind' ) ); ?>">
							<option value="daily" <?php selected( $kind, 'daily' ); ?>><?php esc_html_e( 'Daily', 'freshnews-ai-content-hub' ); ?></option>
							<option value="weekly" <?php selected( $kind, 'weekly' ); ?>><?php esc_html_e( 'Weekly', 'freshnews-ai-content-hub' ); ?></option>
							<option value="monthly" <?php selected( $kind, 'monthly' ); ?>><?php esc_html_e( 'Monthly', 'freshnews-ai-content-hub' ); ?></option>
						</select>
					</p>
					<p>
						<label for="<?php echo esc_attr( $this->get_field_id( 'limit' ) ); ?>"><?php esc_html_e( 'Number of items:', 'freshnews-ai-content-hub' ); ?></label>
						<input class="tiny-text" id="<?php echo esc_attr( $this->get_field_id( 'limit' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'limit' ) ); ?>" type="number" step="1" min="1" max="50" value="<?php echo esc_attr( $limit ); ?>" size="3">
					</p>
					<p>
						<label for="<?php echo esc_attr( $this->get_field_id( 'language' ) ); ?>"><?php esc_html_e( 'Language:', 'freshnews-ai-content-hub' ); ?></label>
						<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'language' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'language' ) ); ?>" type="text" value="<?php echo esc_attr( $language ); ?>" placeholder="<?php echo esc_attr( $default_language ); ?>">
						<small><?php esc_html_e( 'Leave empty to use default language from settings.', 'freshnews-ai-content-hub' ); ?></small>
					</p>
					<?php
				}

				/**
				 * Update widget instance
				 *
				 * @param array $new_instance New instance
				 * @param array $old_instance Old instance
				 * @return array Updated instance
				 */
				public function update( $new_instance, $old_instance ) {
					$instance = array();
					$instance['title'] = ! empty( $new_instance['title'] ) ? sanitize_text_field( $new_instance['title'] ) : '';
					$instance['mode'] = ! empty( $new_instance['mode'] ) ? sanitize_text_field( $new_instance['mode'] ) : 'signals-list';
					$instance['kind'] = ! empty( $new_instance['kind'] ) ? sanitize_text_field( $new_instance['kind'] ) : 'daily';
					$instance['limit'] = ! empty( $new_instance['limit'] ) ? absint( $new_instance['limit'] ) : 5;
					$instance['language'] = ! empty( $new_instance['language'] ) ? preg_replace( '/[^a-z-]/i', '', $new_instance['language'] ) : '';

					return $instance;
				}
			} // End class FreshNews_AI_Widget
		} // End if class_exists WP_Widget
	} // End function
	add_action( 'plugins_loaded', 'freshnews_ai_register_widget_class', 5 );
}
