<?php
/**
 * Plugin Name: FreshNews AI Content Hub
 * Plugin URI: https://freshnews.ai
 * Description: Embed FreshNews.ai content hub directly into your WordPress site using script-based embedding for optimal E-E-A-T SEO value.
 * Version: 1.4.6
 * Author: FreshNews.ai
 * Author URI: https://freshnews.ai
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: freshnews-ai-content-hub
 * Domain Path: /languages
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Define plugin constants
define( 'FRESHNEWS_AI_CONTENT_HUB_VERSION', '1.4.6' );
define( 'FRESHNEWS_AI_CONTENT_HUB_DB_VERSION', '1.0.0' ); // Database/settings schema version
define( 'FRESHNEWS_AI_CONTENT_HUB_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'FRESHNEWS_AI_CONTENT_HUB_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'FRESHNEWS_AI_CONTENT_HUB_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Migration: Derive newsroom_url (Signals URL) from legacy embed_script_url or embed_css_url.
 * Runs once per install. After migration, embed_script_url and embed_css_url are never read for runtime.
 */
function freshnews_ai_migrate_embed_urls_to_signals() {
	if ( get_option( 'freshnews_migrated_embed_urls', 0 ) ) {
		return;
	}
	$options = get_option( 'freshnews_ai_newsroom_options', array() );
	if ( ! is_array( $options ) ) {
		$options = array();
	}
	$signals_url = isset( $options['newsroom_url'] ) ? trim( (string) $options['newsroom_url'] ) : '';
	if ( ! empty( $signals_url ) ) {
		update_option( 'freshnews_migrated_embed_urls', 1 );
		return;
	}
	// Try to derive from embed_script_url
	$embed_script = isset( $options['embed_script_url'] ) ? trim( (string) $options['embed_script_url'] ) : '';
	if ( ! empty( $embed_script ) ) {
		$signals_url = preg_replace( '#/embed/insights\.js/?$#i', '', $embed_script );
		$signals_url = rtrim( $signals_url, '/' );
	}
	// Fallback: derive from embed_css_url
	if ( empty( $signals_url ) && ! empty( $options['embed_css_url'] ) ) {
		$embed_css = trim( (string) $options['embed_css_url'] );
		$signals_url = preg_replace( '#/embed/insights\.css/?$#i', '', $embed_css );
		$signals_url = rtrim( $signals_url, '/' );
	}
	if ( ! empty( $signals_url ) && ( strpos( $signals_url, 'http://' ) === 0 || strpos( $signals_url, 'https://' ) === 0 ) ) {
		$options['newsroom_url'] = $signals_url;
		update_option( 'freshnews_ai_newsroom_options', $options );
	}
	update_option( 'freshnews_migrated_embed_urls', 1 );
}

/**
 * Get normalized Signals base URL (no trailing slash).
 *
 * @return string Empty if not configured or invalid.
 */
function freshnews_get_signals_url() {
	$options = get_option( 'freshnews_ai_newsroom_options', array() );
	$url     = isset( $options['newsroom_url'] ) ? trim( (string) $options['newsroom_url'] ) : '';
	if ( empty( $url ) ) {
		return '';
	}
	$url = rtrim( $url, '/' );
	if ( ! preg_match( '#^https?://#i', $url ) ) {
		$url = 'https://' . $url;
	}
	$url = esc_url_raw( $url );
	if ( empty( $url ) || ( strpos( $url, 'http://' ) !== 0 && strpos( $url, 'https://' ) !== 0 ) ) {
		return '';
	}
	return untrailingslashit( $url );
}

/**
 * Get embed script URL derived from Signals URL.
 *
 * @return string Base + /embed/insights.js, or empty if Signals URL not set.
 */
function freshnews_get_embed_script_url() {
	$base = freshnews_get_signals_url();
	return $base !== '' ? $base . '/embed/insights.js' : '';
}

/**
 * Get embed CSS URL derived from Signals URL.
 *
 * @return string Base + /embed/insights.css, or empty if Signals URL not set.
 */
function freshnews_get_embed_css_url() {
	$base = freshnews_get_signals_url();
	return $base !== '' ? $base . '/embed/insights.css' : '';
}

// Include health checks module (used by admin and ajax)
$health_file = FRESHNEWS_AI_CONTENT_HUB_PLUGIN_DIR . 'includes/health-checks.php';
if ( file_exists( $health_file ) ) {
	require_once $health_file;
}

// Include the main plugin class
$class_file = FRESHNEWS_AI_CONTENT_HUB_PLUGIN_DIR . 'includes/class-freshnews-ai-content-hub.php';
if ( file_exists( $class_file ) ) {
	require_once $class_file;
} else {
	// If class file doesn't exist, show error
	if ( is_admin() ) {
		add_action( 'admin_notices', function() {
			echo '<div class="notice notice-error"><p><strong>FreshNews AI Content Hub:</strong> Plugin class file not found. Please reinstall the plugin.</p></div>';
		} );
	}
	return;
}

/**
 * Activation hook
 */
function freshnews_ai_content_hub_activate() {
	// Set default options (option key kept for backward compatibility with freshnews_ai_newsroom)
	$defaults = array(
		'enabled'              => true,
		'newsroom_url'         => '',
		'page_slug'            => 'signals',
		'page_title'           => 'Signals',
		'enable_insights_page' => false,
		'insights_page_slug'   => 'insights',
		'insights_page_title'  => 'Insights',
		'enable_faq_page'      => false,
		'language'             => 'en',
		'theme'                => 'auto',
		'canonical_mode'       => 'use_newsroom_url',
		'auto_create_page'     => true,
		'defer_script_loading' => true,
		'widget_mode'          => 'signals-list',
		'widget_kind'          => 'daily',
		'widget_limit'         => 20,
		'link_base'            => '',
		'footer_pulse_enabled'   => false,
		'footer_pulse_label'    => 'Latest Signal',
		'footer_pulse_lang_mode' => 'auto',
		'footer_pulse_lang'     => 'en',
		'enable_robots_sitemaps' => true,
		'enable_llms_txt'       => false,
	);

	// Only set defaults if option doesn't exist
	if ( ! get_option( 'freshnews_ai_newsroom_options' ) ) {
		add_option( 'freshnews_ai_newsroom_options', $defaults );
	}

	// Flush rewrite rules to register article URL patterns
	flush_rewrite_rules();

	// Set database version
	update_option( 'freshnews_ai_newsroom_db_version', FRESHNEWS_AI_CONTENT_HUB_DB_VERSION );
}
register_activation_hook( __FILE__, 'freshnews_ai_content_hub_activate' );

/**
 * Check for plugin updates and run migrations
 */
function freshnews_ai_content_hub_check_updates() {
	$current_db_version = get_option( 'freshnews_ai_newsroom_db_version', '0.0.0' );
	$plugin_db_version  = FRESHNEWS_AI_CONTENT_HUB_DB_VERSION;

	// Only run migrations if versions differ
	if ( version_compare( $current_db_version, $plugin_db_version, '<' ) ) {
		freshnews_ai_content_hub_run_migrations( $current_db_version, $plugin_db_version );
		update_option( 'freshnews_ai_newsroom_db_version', $plugin_db_version );
	}
}

/**
 * Run database migrations
 *
 * @param string $from_version Current database version
 * @param string $to_version   Target database version
 */
function freshnews_ai_content_hub_run_migrations( $from_version, $to_version ) {
	// Migration from 0.0.0 to 1.0.0 (initial setup)
	if ( version_compare( $from_version, '1.0.0', '<' ) ) {
		// Ensure default options exist
		$options  = get_option( 'freshnews_ai_newsroom_options', array() );
		$defaults = array(
			'enabled'              => true,
			'newsroom_url'         => '',
			'page_slug'            => 'signals',
			'page_title'           => 'Signals',
			'enable_insights_page' => false,
			'insights_page_slug'   => 'insights',
			'insights_page_title'  => 'Insights',
			'enable_faq_page'      => false,
			'language'             => 'en',
			'theme'                => 'auto',
			'canonical_mode'       => 'use_newsroom_url',
			'auto_create_page'     => true,
			'defer_script_loading' => true,
			'widget_mode'          => 'signals-list',
			'widget_kind'          => 'daily',
		'widget_limit'         => 20,
		'link_base'            => '',
		'footer_pulse_enabled'   => false,
		'footer_pulse_label'    => 'Latest Signal',
		'footer_pulse_lang_mode' => 'auto',
		'footer_pulse_lang'     => 'en',
		'enable_robots_sitemaps' => true,
		'enable_llms_txt'       => false,
	);

		// Merge with existing options (preserve user settings)
		$options = wp_parse_args( $options, $defaults );

		// Ensure new insights page fields have defaults if not set
		if ( ! isset( $options['enable_insights_page'] ) ) {
			$options['enable_insights_page'] = false;
		}
		if ( ! isset( $options['insights_page_slug'] ) ) {
			$options['insights_page_slug'] = 'insights';
		}
		if ( ! isset( $options['insights_page_title'] ) ) {
			$options['insights_page_title'] = 'Insights';
		}
		if ( ! isset( $options['enable_faq_page'] ) ) {
			$options['enable_faq_page'] = false;
		}

		// Ensure footer pulse defaults (migrate from legacy keys if needed)
		if ( ! isset( $options['footer_pulse_enabled'] ) ) {
			$options['footer_pulse_enabled'] = ! empty( $options['auto_footer_latest'] );
		}
		if ( ! isset( $options['footer_pulse_label'] ) ) {
			$options['footer_pulse_label'] = isset( $options['footer_label_en'] ) ? $options['footer_label_en'] : 'Latest Signal';
		}
		if ( ! isset( $options['footer_pulse_lang_mode'] ) ) {
			$options['footer_pulse_lang_mode'] = 'auto';
		}
		if ( ! isset( $options['footer_pulse_lang'] ) ) {
			$options['footer_pulse_lang'] = 'en';
		}
		if ( ! isset( $options['enable_robots_sitemaps'] ) ) {
			// Default true (recommended); preserve legacy: was tied to enabled, so migrate accordingly
			$options['enable_robots_sitemaps'] = ! empty( $options['enabled'] );
		}
		if ( ! isset( $options['enable_llms_txt'] ) ) {
			$options['enable_llms_txt'] = false;
		}

		update_option( 'freshnews_ai_newsroom_options', $options );
	}

	// Future migrations can be added here
}

/**
 * Initialize the plugin
 */
function freshnews_ai_content_hub_init() {
	// Migrate legacy embed URLs to Signals URL (runs once)
	freshnews_ai_migrate_embed_urls_to_signals();

	// Check for updates and run migrations
	freshnews_ai_content_hub_check_updates();

	// Check if class exists
	if ( ! class_exists( 'FreshNews_AI_Content_Hub' ) ) {
		if ( is_admin() ) {
			add_action( 'admin_notices', function() {
				echo '<div class="notice notice-error"><p><strong>FreshNews AI Content Hub:</strong> Plugin class not found. Please deactivate and reactivate the plugin.</p></div>';
			} );
		}
		return;
	}

	$GLOBALS['freshnews_ai_content_hub'] = new FreshNews_AI_Content_Hub();
	$GLOBALS['freshnews_ai_content_hub']->init();
}
add_action( 'plugins_loaded', 'freshnews_ai_content_hub_init' );
