/**
 * FreshNews.ai Embed Loader
 *
 * This script initializes the FreshNews.ai embed widget by:
 * 1. Finding the #freshnews-ai-root container
 * 2. Reading data attributes from the container
 * 3. Setting global configuration objects for the embed script
 * 4. Ensuring the external embed script is loaded only once
 */

(function() {
	'use strict';

	// Prevent multiple initializations
	if (window.freshnewsAiEmbedLoaderInitialized) {
		return;
	}
	window.freshnewsAiEmbedLoaderInitialized = true;

	/**
	 * Fetch and render article
	 */
	function fetchArticleAndRender(container, apiBase, slug) {
		// Show loading state
		container.innerHTML = '<div class="fn-embed fn-embed-article"><div class="fn-embed-loading">Loading article...</div></div>';

		// Try to access renderArticle from the embed script's scope
		// The embed script might expose it globally or we can try to trigger it via events
		// First, try to find if renderArticle is available in any scope
		var renderArticleFn = null;

		// Check if it's on window (unlikely but possible)
		if (typeof window.renderArticle === 'function') {
			window.renderArticle(container, apiBase, null, slug, true);
			return;
		}

		// Try to trigger the embed script's navigation handler
		// The embed script might listen for certain events or we can simulate a click on a hub domain link
		// For now, let's try to manually call the embed script's logic
		// But since we can't easily access it, we'll fetch and render manually

		// Fallback: Fetch article data and render manually
		// Use the same logic as the embed script's fetchArticleData function
		var apiUrl = apiBase.replace(/\/+$/, '');
		var language = container.getAttribute('data-language') || 'en';
		var tenant = container.getAttribute('data-tenant') || null;
		var articleUrl;

		// Extract domain from apiBase (same logic as embed script)
		var domain = null;
		var domainMatch = apiBase.match(/https?:\/\/([^\/:]+)/);
		if (domainMatch) {
			domain = domainMatch[1].split(':')[0]; // Remove port if present
		}

		// Use domain-based endpoint if domain is available (same as embed script)
		// Note: The embed script uses "/en/articles/" hardcoded, but we'll use the language from container
		if (domain) {
			// Use domain-based endpoint: /api/public/[domain]/[lang]/articles/[slug]
			articleUrl = apiUrl + '/api/public/' + encodeURIComponent(domain) + '/' + encodeURIComponent(language) + '/articles/' + encodeURIComponent(slug);
		} else if (tenant) {
			// Fallback: Use tenantSlug-based endpoint
			articleUrl = apiUrl + '/api/public/tenants/' + encodeURIComponent(tenant) + '/articles/' + encodeURIComponent(slug);
		} else {
			// No domain and no tenant - show error
			container.innerHTML = '<div class="fn-embed fn-embed-error">Configuration error: Must provide either a custom domain (CNAME) or tenant slug.</div>';
			return;
		}

		// Fetch article (same headers as embed script)
		fetch(articleUrl, {
			headers: {
				'Accept': 'application/json'
			}
		})
			.then(function(response) {
				if (!response.ok) {
					throw new Error('Failed to fetch article: ' + response.status + ' ' + response.statusText);
				}
				return response.json();
			})
			.then(function(data) {
				// The API might return { article: {...}, persona: {...} } or FrontendArticleV1 directly
				// Same logic as embed script
				var article = data.article || data;
				if (!article) {
					throw new Error('Article data not found in API response');
				}

				// Render article HTML (simplified version)
				var html = '<div class="fn-embed fn-embed-article">';
				html += '<button class="fn-embed-back" onclick="window.history.back()">← Back</button>';

				if (article.hero && article.hero.title) {
					html += '<h1 class="fn-embed-title">' + escapeHtml(article.hero.title) + '</h1>';
				}
				if (article.hero && article.hero.dek) {
					html += '<div class="fn-embed-dek">' + escapeHtml(article.hero.dek) + '</div>';
				}
				if (article.hero && article.hero.image && article.hero.image.url) {
					// Wrap image in div with proper class (matches embed script structure)
					html += '<div class="fn-embed-hero">';
					html += '<img src="' + escapeHtml(article.hero.image.url) + '" alt="' + escapeHtml(article.hero.image.alt || article.hero.title || '') + '" loading="lazy" decoding="async" />';
					html += '</div>';
				}
				if (article.sections) {
					article.sections.forEach(function(section) {
						if (section.heading) {
							html += '<h2>' + escapeHtml(section.heading) + '</h2>';
						}
						if (section.blocks) {
							section.blocks.forEach(function(block) {
								if (block.type === 'p') {
									html += '<p>' + escapeHtml(block.text) + '</p>';
								} else if (block.type === 'ul' && block.items) {
									html += '<ul>';
									block.items.forEach(function(item) {
										html += '<li>' + escapeHtml(item) + '</li>';
									});
									html += '</ul>';
								}
							});
						}
					});
				}
				html += '</div>';

				container.innerHTML = html;
			})
			.catch(function(error) {
				console.error('[FreshNews Embed Loader] Failed to fetch article:', error);
				container.innerHTML = '<div class="fn-embed fn-embed-error">Failed to load article. <a href="' + articleUrl + '">Open in new tab</a></div>';
			});
	}

	/**
	 * Escape HTML
	 */
	function escapeHtml(text) {
		var div = document.createElement('div');
		div.textContent = text;
		return div.innerHTML;
	}

	/**
	 * Extract slug from URL path
	 */
	function extractSlugFromPath(url, linkBase) {
		try {
			var urlObj = new URL(url, window.location.href);
			var path = urlObj.pathname;

			// If linkBase is set, try to extract slug relative to it
			if (linkBase) {
				var basePath = linkBase.replace(/^https?:\/\/[^\/]+/, '').replace(/\/$/, '');
				if (path.startsWith(basePath + '/')) {
					var slug = path.substring(basePath.length + 1);
					// Remove trailing slash
					slug = slug.replace(/\/$/, '');
					if (slug) return slug;
				}
			}

			// Fallback: get last path segment
			var parts = path.split('/').filter(function(p) { return p; });
			if (parts.length > 0) {
				return parts[parts.length - 1];
			}

			return null;
		} catch (e) {
			return null;
		}
	}

	/**
	 * Setup click interception for content hub domain links (fallback)
	 * This ensures clicks are intercepted even if embed script doesn't handle it
	 */
	function setupNewsroomLinkInterception(root) {
		var pageBase = root.getAttribute('data-page-base') || '';
		var apiBase = root.getAttribute('data-api-base') || '';
		var language = root.getAttribute('data-language') || 'en';

		if (!pageBase || !apiBase) return;

		// Intercept clicks on content hub domain links
		root.addEventListener('click', function(e) {
			var link = e.target.closest('a.fn-embed-link, a.fn-embed-card-link');
			if (!link) return;

			var href = link.getAttribute('href');
			if (!href) return;

			try {
				var linkUrl = new URL(href, window.location.href);
				var apiUrl = new URL(apiBase);

				// Only intercept links pointing to content hub domain
				if (linkUrl.hostname === apiUrl.hostname) {
					e.preventDefault();
					e.stopPropagation();

					// Extract slug from URL
					var slug = extractSlugFromUrl(href, apiBase);
					if (!slug) {
						// Can't extract slug, allow normal navigation
						window.location.href = href;
						return;
					}

					// Build new path using page-base
					var basePath = pageBase.replace(/\/$/, '');
					var newPath;

					if (language && language !== 'en') {
						newPath = '/' + encodeURIComponent(language) + basePath + '/' + encodeURIComponent(slug);
					} else {
						newPath = basePath + '/' + encodeURIComponent(slug);
					}

					// Update URL with pushState
					var currentPath = window.location.pathname;
					if (currentPath !== newPath) {
						window.history.pushState({ article: slug, path: newPath }, '', newPath);
					}

					// Try to trigger embed script's renderArticle
					// First, try to find and call it directly
					if (typeof window.renderArticle === 'function') {
						window.renderArticle(root, apiBase, null, slug, true);
					} else {
						// Fallback: fetch and render manually
						fetchArticleAndRender(root, apiBase, slug);
					}
				}
			} catch (err) {
				// If URL parsing fails, allow normal navigation
				console.warn('[FreshNews Embed Loader] Could not parse URL:', href, err);
			}
		}, true); // Use capture phase to catch clicks early

		// Handle back/forward navigation
		window.addEventListener('popstate', function(e) {
			if (e.state && e.state.article) {
				var slug = e.state.article;

				// Try to trigger embed script's renderArticle
				if (typeof window.renderArticle === 'function') {
					window.renderArticle(root, apiBase, null, slug, true);
				} else {
					// Fallback: fetch and render manually
					fetchArticleAndRender(root, apiBase, slug);
				}
			} else {
				// No article in state - show list view
				// The embed script should handle this, but we can trigger a reload
				window.location.reload();
			}
		});
	}

	/**
	 * Extract slug from content hub URL
	 */
	function extractSlugFromUrl(url, apiBase) {
		try {
			var urlObj = new URL(url, window.location.href);

			// Check for query parameters first
			var slug = urlObj.searchParams.get('slug') || urlObj.searchParams.get('article');
			if (slug) return slug;

			// Check path format: /tenant/signals/slug or /en/signals/slug or /signals/slug
			var pathParts = urlObj.pathname.split('/').filter(function(p) { return p; });

			// Look for slug after 'signals' or 'insights'
			var signalsIndex = pathParts.indexOf('signals');
			var insightsIndex = pathParts.indexOf('insights');
			var index = signalsIndex !== -1 ? signalsIndex : insightsIndex;

			if (index !== -1 && index < pathParts.length - 1) {
				return pathParts[index + 1];
			}

			// Fallback: last part
			if (pathParts.length > 0) {
				var lastPart = pathParts[pathParts.length - 1];
				// Skip common paths
				var skipParts = ['signals', 'insights', 'about', 'en', 'he', 'es', 'fr', 'de', 'it', 'pt', 'ja', 'zh', 'ar', 'null'];
				if (!skipParts.includes(lastPart) && lastPart.length > 0) {
					return lastPart;
				}
			}

			return null;
		} catch (e) {
			return null;
		}
	}

	/**
	 * Verify embed navigation is set up correctly
	 */
	function verifyEmbedNavigation(root) {
		var pageBase = root.getAttribute('data-page-base');
		var apiBase = root.getAttribute('data-api-base');

		if (!pageBase || !apiBase) return;

		// Check if we're in embed context (different domain)
		try {
			var apiUrl = new URL(apiBase);
			var isEmbedContext = window.location.hostname !== apiUrl.hostname;

			if (isEmbedContext && pageBase) {
				// Log for debugging
				if (window.location.hostname === 'localhost' || window.location.hostname.includes('localhost') || window.location.hostname.includes('.local')) {
					console.log('[FreshNews Embed Loader] Embed context detected:', {
						currentHost: window.location.hostname,
						apiHost: apiUrl.hostname,
						pageBase: pageBase,
						isEmbedContext: isEmbedContext
					});
				}
			}
		} catch (e) {
			// Ignore
		}
	}

	/**
	 * Setup click interception for WordPress domain links
	 * NOTE: This is no longer needed if we let the embed script handle everything
	 * The embed script intercepts content hub domain links, and page-base keeps URLs on WordPress domain
	 * Keeping this function for potential future use, but it won't be called if link-base is not set
	 */
	function setupWordPressLinkInterception(root) {
		var linkBase = root.getAttribute('data-link-base') || '';
		var pageBase = root.getAttribute('data-page-base') || '';
		var apiBase = root.getAttribute('data-api-base') || '';

		// If link-base is not set or points to same domain as api-base, embed script handles it
		if (!linkBase || !apiBase) return;

		try {
			var linkBaseUrl = new URL(linkBase, window.location.href);
			var apiBaseUrl = new URL(apiBase);

			// If link-base and api-base are on the same domain, embed script will handle it
			if (linkBaseUrl.hostname === apiBaseUrl.hostname) {
				return; // No need to intercept, embed script handles it
			}

			// If different domains, we'd need to intercept, but this causes CORS issues
			// So we don't set link-base in this case, letting embed script handle everything
			return;
		} catch (e) {
			return;
		}

		// Intercept clicks on links within the embed container
		root.addEventListener('click', function(e) {
			var link = e.target.closest('a.fn-embed-link, a.fn-embed-card-link');
			if (!link) return;

			var href = link.getAttribute('href');
			if (!href) return;

			try {
				var linkUrl = new URL(href, window.location.href);
				var linkBaseUrl = new URL(linkBase, window.location.href);

				// Only intercept links on the WordPress domain (link-base domain)
				if (linkUrl.hostname === linkBaseUrl.hostname) {
					e.preventDefault();
					e.stopPropagation();

					// Extract slug from URL
					var slug = extractSlugFromPath(href, linkBase);
					if (!slug) {
						// Can't extract slug, allow normal navigation
						window.location.href = href;
						return;
					}

					// Build new path using page-base
					var basePath = pageBase || linkBaseUrl.pathname.replace(/\/$/, '') || '/signals';
					var newPath;

					if (language && language !== 'en') {
						newPath = '/' + encodeURIComponent(language) + basePath + '/' + encodeURIComponent(slug);
					} else {
						newPath = basePath + '/' + encodeURIComponent(slug);
					}

					// Update URL with pushState
					var currentPath = window.location.pathname;
					if (currentPath !== newPath) {
						window.history.pushState({ article: slug, path: newPath }, '', newPath);
					}

					// Try to access renderArticle from the embed script
					// The embed script might expose it globally or we need to trigger it differently
					// For now, we'll manually fetch and render the article
					fetchArticleAndRender(root, apiBase, slug);
				}
			} catch (err) {
				// If URL parsing fails, allow normal navigation
				console.warn('[FreshNews Embed Loader] Could not parse URL:', href, err);
			}
		}, true); // Use capture phase

		// Handle back/forward navigation
		window.addEventListener('popstate', function(e) {
			if (e.state && e.state.article) {
				var slug = e.state.article;
				var apiBase = root.getAttribute('data-api-base') || '';

				// Fetch and render article
				fetchArticleAndRender(root, apiBase, slug);
			}
		});
	}

	/**
	 * Initialize the embed
	 */
	function initEmbed() {
		// Find the root container (may or may not be wrapped, depending on context)
		var root = document.getElementById('freshnews-ai-root');
		if (!root) {
			return;
		}

		// Check if we're in WordPress embed context (different domain than api-base)
		// Only set up WordPress-specific handling if needed
		var apiBase = root.getAttribute('data-api-base') || '';
		var isWordPressEmbed = false;

		if (apiBase) {
			try {
				var apiUrl = new URL(apiBase);
				isWordPressEmbed = window.location.hostname !== apiUrl.hostname;
			} catch (e) {
				// Can't determine, assume regular embed
			}
		}

		// Only set up WordPress-specific features if we're in WordPress embed context
		if (isWordPressEmbed) {
			// Verify the wrapper has data-embed-nav (should be set by PHP)
			var wrapper = root.parentElement;
			if (wrapper && !wrapper.getAttribute('data-embed-nav')) {
				// Ensure it's set (fallback)
				wrapper.setAttribute('data-embed-nav', 'true');
			}
		}

		// Get configuration from WordPress (passed via wp_localize_script)
		var config = window.freshnewsAiEmbedConfig || {};

		// Store config for use in checkAndLoadInsightsScript
		if (!window.freshnewsAiEmbedConfig) {
			window.freshnewsAiEmbedConfig = config;
		}

		// Read data attributes from the container (these override config)
		var newsroomUrl = root.getAttribute('data-api-base') || config.newsroomUrl || '';
		var locale = root.getAttribute('data-language') || root.getAttribute('data-locale') || config.locale || 'en';
		var theme = root.getAttribute('data-theme') || config.theme || 'auto';
		var canonicalUrlMode = config.canonicalUrlMode || 'use_newsroom_url';

		// Build configuration object
		var embedConfig = {
			newsroomUrl: newsroomUrl,
			locale: locale,
			theme: theme,
			canonicalUrlMode: canonicalUrlMode
		};

		// Set global config objects for compatibility
		// Support both naming conventions
		window.FRESHNEWS_AI_EMBED = embedConfig;
		window.freshnewsAiEmbedConfig = embedConfig;

		// Only set up WordPress-specific features if we're in WordPress embed context
		if (isWordPressEmbed) {
			// Setup click interception for WordPress domain links
			// This handles the case where link-base points to WordPress domain
			setupWordPressLinkInterception(root);

			// Also intercept content hub domain links as fallback
			// The embed script should handle this, but we'll add a fallback to ensure it works
			setupNewsroomLinkInterception(root);

			// Also ensure embed script intercepts content hub domain links and uses page-base
			// Wait for embed script to initialize, then verify it's set up correctly
			setTimeout(function() {
				verifyEmbedNavigation(root);
			}, 500);
		}

		// The external embed script should already be loaded via wp_enqueue_script
		// Verify it loaded and manually trigger if needed
		// Also handle case where script fails to load (CORS, network issues, etc.)
		var insightsRetryCount = 0;
		var INSIGHTS_MAX_RETRIES = 8;

		function checkAndLoadInsightsScript() {
			var containers = document.querySelectorAll('[data-freshnews-widget="insights"]');
			var container = containers.length > 0 ? containers[0] : null;

			// Check if initFreshNewsInsights is available (means insights.js loaded)
			if (typeof window.initFreshNewsInsights === 'function') {
				// Manually trigger initFreshNewsInsights to ensure it runs
				try {
					window.initFreshNewsInsights();
				} catch (e) {
					console.error('[FreshNews Embed Loader] Error calling initFreshNewsInsights:', e);
				}
				return true;
			}

			// Script not loaded - stop retrying after max attempts
			if (insightsRetryCount >= INSIGHTS_MAX_RETRIES) {
				if (window.location.hostname === 'localhost' || window.location.hostname.includes('localhost') || window.location.hostname.includes('.local')) {
					console.warn('[FreshNews Embed Loader] insights.js did not load after', INSIGHTS_MAX_RETRIES, 'attempts. Check Embed Script URL in WordPress settings and ensure', document.querySelector('script[src*="/embed/insights"]') ? 'the script loads without errors.' : 'the Signals Hub is reachable.');
				}
				return false;
			}
			insightsRetryCount += 1;

			// Try to load it dynamically
			var embedConfig = window.freshnewsAiEmbedConfig || config || {};
			var embedScriptUrl = embedConfig.embedScriptUrl || '';
			if (!embedScriptUrl && container) {
				var apiBase = container.getAttribute('data-api-base') || '';
				if (apiBase) {
					embedScriptUrl = apiBase.replace(/\/+$/, '') + '/embed/insights.js';
				}
			}

			if (embedScriptUrl) {
				// Check if script is already in the DOM (WP may add ?ver=, so match by /embed/insights)
				var existingScript = document.querySelector('script[src*="/embed/insights"]');

				if (existingScript) {
					// Script tag exists but function not available - retry with cap
					setTimeout(checkAndLoadInsightsScript, 1000);
					return false;
				}

				// Dynamically load the script (only once to avoid duplicate tags)
				if (insightsRetryCount === 1) {
					var script = document.createElement('script');
					script.src = embedScriptUrl;
					script.async = true;
					script.onload = function() {
						setTimeout(checkAndLoadInsightsScript, 100);
					};
					script.onerror = function() {
						console.error('[FreshNews Embed Loader] Failed to load insights.js from:', embedScriptUrl, '- check Embed Script URL and CORS.');
						setTimeout(checkAndLoadInsightsScript, 1000);
					};
					document.head.appendChild(script);
				} else {
					setTimeout(checkAndLoadInsightsScript, 1000);
				}
				return false;
			}

			setTimeout(checkAndLoadInsightsScript, 1000);
			return false;
		}

		// Check after short delay, then retry with cap
		setTimeout(checkAndLoadInsightsScript, 500);
	}

	// Initialize when DOM is ready
	if (document.readyState === 'loading') {
		document.addEventListener('DOMContentLoaded', initEmbed);
	} else {
		// DOM is already ready
		initEmbed();
	}

	// Also try to initialize after a short delay in case the script loads before the container
	setTimeout(initEmbed, 100);
})();
